/*
 * 08/03/2009, 14:07.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.util;

import java.util.regex.Pattern;

/**
 * Esta classe representa um pacote de funes auxiliares para manipular
 * {@linkplain String}s.
 * @author Victor Williams Stafusa da Silva
 */
public final class StringUtils {

    /**
     * Esta classe no  instancivel.
     */
    private StringUtils() {}

    /**
     * Verifica se uma {@linkplain String} contm algum contedo.
     * @param s A {@linkplain String} a ser verificada.
     * @return {@code true} se a {@linkplain String} no tiver contedo, ou for
     * {@code null}, {@code false} em caso contrrio.
     */
    public static boolean vazio(String s) {
        return s == null || "".equals(s);
    }

    /**
     * Compara duas {@linkplain String}s de modo case-sensitive, mesmo se uma
     * delas ou as duas forem iguais a {@code null}.
     * @param s1 Uma das {@linkplain String}s a ser comparada.
     * @param s2 A outra {@linkplain String} a ser comparada.
     * @return {@code true} se ambas as {@linkplain String}s forem idnticas ou
     * se ambas forem {@code null}, {@code false} em caso contrrio.
     */
    public static boolean equals(String s1, String s2) {
        return (s1 == null && s2 == null) || (s1 != null && s1.equals(s2));
    }

    /**
     * Usado internamente para fazer a converso rapidamente de um nmero para o
     * dgito hexadecimal correspondente.
     */
    private static final char[] HEX = "0123456789ABCDEF".toCharArray();

    /**
     * Faz o escape de um {@code char} para o formato JSON.
     * @param c O {@code char} a ser escapada.
     * @return A {@linkplain String} escapada.
     */
    public static String jsonFormat(char c) {
        if (c == '\"') {
            return "\\\"";
        } else if (c == '\\') {
            return "\\\\";
        } else if (c == '\n') {
            return "\\n";
        } else if (c < ' ' || c >= 127) {
            return new StringBuilder("\\u")
                    .append(HEX[(c & 0xf000) >> 12])
                    .append(HEX[(c & 0x0f00) >> 8])
                    .append(HEX[(c & 0x00f0) >> 4])
                    .append(HEX[(c & 0x000f)])
                    .toString();
        } else {
            return String.valueOf(c);
        }
    }

    /**
     * Faz o escape de uma {@linkplain String} para o formato JSON. Caso a
     * {@linkplain String} informada seja {@code null}, ento uma {@linkplain
     * String} vazia  retornada.
     * @param original A {@linkplain String} a ser escapada.
     * @return A {@linkplain String} escapada.
     */
    public static String jsonFormat(String original) {
        if (original == null) return "";

        StringBuilder sb = new StringBuilder();
        for (char c : original.toCharArray()) {
            sb.append(jsonFormat(c));
        }
        return sb.toString();
    }

    /**
     * {@linkplain Pattern} usado internamente para determinar se um endereo de
     * e-mail  vlido ou no.
     */
    private static final Pattern EMAIL_PATTERN =
            Pattern.compile("[a-z0-9_]+([\\.][a-z0-9_]+)*@[a-z0-9_]+([\\.][a-z0-9_]+)+");

    /**
     * Determina se um endereo de e-mail informado constitui um endereo
     * vlido (mas no necessariamente existente).
     * @param s O endereo de e-mail a ser testado.
     * @return {@code true} se o endereo de e-mail informado  vlido,
     * {@code false} em caso contrrio.
     */
    public static boolean emailValido(String s) {
        return s != null && EMAIL_PATTERN.matcher(s).matches();
    }
}
