/*
 * 18/03/2009, 20:50.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.usuario;

import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;

/**
 * Classe responsvel por representar a funcionalidade de visualizao de perfil
 * de usurio, contendo alm do perfil encontrado, os erros que possam ter
 * ocorrido na busca do perfil.
 * @author Adson Carlos Nery Malta
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public class FuncionalidadeVisualizarUsuario implements Serializable {

    private static final long serialVersionUID = -6877089273083524866L;

    /** O id do usurio a ser visualizado. */
    private final Long idUsuario;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** O bean de usurio localizado. */
    private final Usuario usuario;

    public enum TiposErros { ERRO }

    /**
     * Realiza a busca do perfil de um usurio.
     * @param idUsuario O id do usurio a ser visualizado.
     */
    public FuncionalidadeVisualizarUsuario(Long idUsuario) {
        this.idUsuario = idUsuario;
        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);
        
        if (idUsuario == null) {
            problemas.adicionarProblema(TiposErros.ERRO, "O usurio deve ser informado.");
        }

        // Pesquisa o usurio.
        Usuario u = null;
        if (problemas.isVazio()) {
            try {
                u = Usuario.pesquisarPorId(idUsuario);
                if (u == null) {
                    problemas.adicionarProblema(TiposErros.ERRO, "Esse usurio no existe.");
                }
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }
        usuario = u;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    /**
     * Obtm o id do usurio a ser visualizado.
     * @return O id do usurio a ser visualizado.
     */
    @JsonMarshal
    public Long getIdUsuario() {
        return idUsuario;
    }

    /**
     * Obtm o bean de usurio localizado, ou {@code null} caso no tenha sido
     * encontrado.
     * @return O bean de usurio localizado.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de obteno do perfil
     * de usurio.
     * @return Os problemas resultantes do processo de obteno do perfil de
     * usurio.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }

    /**
     * Indica se o processo de obteno do perfil de usurio foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }
}