/*
 * 13/03/2009, 20:16.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.questao;

import br.com.simuquiz.entidades.Exame;
import br.com.simuquiz.entidades.Questao;
import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import static br.com.simuquiz.util.StringUtils.*;

/**
 * @author Adson Carlos Nery Malta
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeCadastrarQuestao implements Serializable {

    private static final long serialVersionUID = -2450524832667753748L;

    /**  */
    private final Usuario usuario;

    /**  */
    private final String enunciado;

    /**  */
    private final List<String> alternativas;

    private final Long idExame;

    private final Long idQuestao;

    /**  */
    private final Set<Integer> corretas;

    /**  */
    private final Integer maxCorretas;

    /**  */
    private final Integer minCorretas;

    /**  */
    private final Integer maxAlternativas;

    /**  */
    private final Integer minAlternativas;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    private final Questao questao;

    public enum TiposErros {
        ERRO, ENUNCIADO, ALTERNATIVAS, MAX_CORRETAS, MIN_CORRETAS, MAX_ALTERNATIVAS, MIN_ALTERNATIVAS
    }

    public static FuncionalidadeCadastrarQuestao incluir(Usuario usuario, Long idExame, String enunciado, List<String> alternativas, Set<Integer> corretas, Integer maxCorretas, Integer minCorretas, Integer maxAlternativas, Integer minAlternativas) {
        return new FuncionalidadeCadastrarQuestao(usuario, null, idExame, enunciado, alternativas, corretas, maxCorretas, minCorretas, maxAlternativas, minAlternativas);
    }

    public static FuncionalidadeCadastrarQuestao alterar(Usuario usuario, Long idQuestao, Long idExame, String enunciado, List<String> alternativas, Set<Integer> corretas, Integer maxCorretas, Integer minCorretas, Integer maxAlternativas, Integer minAlternativas) {
        return new FuncionalidadeCadastrarQuestao(usuario, idQuestao, idExame, enunciado, alternativas, corretas, maxCorretas, minCorretas, maxAlternativas, minAlternativas);
    }

    private FuncionalidadeCadastrarQuestao(Usuario usuario, Long idQuestao, Long idExame, String enunciado, List<String> alternativas, Set<Integer> corretas, Integer maxCorretas, Integer minCorretas, Integer maxAlternativas, Integer minAlternativas) {
        if (usuario == null) {
            throw new IllegalArgumentException();
        }

        // Grava os parmetros informados.
        this.enunciado = enunciado;
        this.idExame = idExame;
        this.idQuestao = idQuestao;
        alternativas = (alternativas == null ? Collections.<String>emptyList() : alternativas);
        this.alternativas = alternativas;
        corretas = (corretas == null ? Collections.<Integer>emptySet() : corretas);
        this.corretas = corretas;
        this.maxCorretas = maxCorretas;
        this.minCorretas = minCorretas;
        this.maxAlternativas = maxAlternativas;
        this.minAlternativas = minAlternativas;
        this.usuario = usuario;

        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);

        // Validaes bsicas.
        if (idExame == null) {
            problemas.adicionarProblema(TiposErros.ERRO, "O exame deve ser informado.");
        }
        if (vazio(enunciado)) {
            problemas.adicionarProblema(TiposErros.ENUNCIADO, "O enunciado da questo deve ser preenchido.");
        }

        // Valida se todas as alternativas foram preenchidas e limpa os espaos
        // em excesso delas.
        boolean existemAlternativasVazias = false;
        List<String> alternativasLimpo = new ArrayList<String>(alternativas.size());
        for (String s : alternativas) {
            if (vazio(s)) {
                if (!existemAlternativasVazias) {
                    problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "O texto de todas as alternativas deve ser preenchido.");
                    existemAlternativasVazias = true;
                }
                alternativasLimpo.add("");
            } else {
                alternativasLimpo.add(s.trim());
            }
        }

        // Verifica se h alternativas com textos duplicados.
        int indiceProximo = 0;
        for (String s : alternativasLimpo) {
            indiceProximo++;
            if (indiceProximo > alternativasLimpo.size() - 1) continue;
            for (String t : alternativasLimpo.subList(indiceProximo, alternativasLimpo.size() - 1)) {
                if (!t.equals(s)) continue;
                problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "Os textos das alternativas no podem se repetir.");
                break;
            }
        }

        // Validaes no nmero de alternativas.
        if (alternativas.size() < 2) {
            problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "A questo deve ter pelo menos 2 alternativas.");
        }
        if (corretas.size() == 0) {
            problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "Deve haver pelo menos uma alternativa verdadeira.");
        } else {
            for (Integer i : corretas) {
                if (i >= 0 && i < alternativas.size()) continue;
                problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "A alternativa no existente " + i + " foi marcada como correta.");
            }
        }
        if (corretas.size() == alternativas.size()) {
            problemas.adicionarProblema(TiposErros.ALTERNATIVAS, "Deve haver pelo menos uma alternativa falsa.");
        }
        if (minCorretas == null) {
            problemas.adicionarProblema(TiposErros.MIN_CORRETAS, "O nmero mnimo de alternativas corretas deve ser especificado.");
        }
        if (minCorretas != null && minCorretas < 1) {
            problemas.adicionarProblema(TiposErros.MIN_CORRETAS, "O nmero mnimo de alternativas corretas deve ser maior ou igual a 1.");
        }
        if (maxCorretas == null) {
            problemas.adicionarProblema(TiposErros.MAX_CORRETAS, "O nmero mximo de alternativas corretas deve ser especificado.");
        }
        if (maxCorretas != null && maxCorretas > corretas.size()) {
            problemas.adicionarProblema(TiposErros.MAX_CORRETAS, "O nmero mximo de alternativas corretas no pode ser superior ao nmero total de alternativas corretas.");
        }
        if (minCorretas != null && maxCorretas != null && maxCorretas < minCorretas) {
            problemas.adicionarProblema(TiposErros.MAX_CORRETAS, "O nmero mximo de alternativas corretas no pode ser inferior ao mnimo.");
        }
        if (minAlternativas == null) {
            problemas.adicionarProblema(TiposErros.MIN_ALTERNATIVAS, "O nmero mnimo de alternativas na questo deve ser preenchido.");
        }
        if (minAlternativas != null && minAlternativas < 2) {
            problemas.adicionarProblema(TiposErros.MIN_ALTERNATIVAS, "O nmero mnimo de alternativas na questo deve ser maior ou igual a 2.");
        }
        if (maxAlternativas == null) {
            problemas.adicionarProblema(TiposErros.MAX_ALTERNATIVAS, "O nmero mximo de alternativas na questo deve ser preenchido.");
        }
        if (maxAlternativas != null && maxAlternativas > alternativas.size()) {
            problemas.adicionarProblema(TiposErros.MAX_ALTERNATIVAS, "O nmero mximo de alternativas na questo no pode ser superior ao nmero total de alternativas.");
        }
        if (minAlternativas != null && maxAlternativas != null && maxAlternativas < minAlternativas) {
            problemas.adicionarProblema(TiposErros.MAX_ALTERNATIVAS, "O nmero mximo de alternativas na questo no pode ser inferior ao mnimo.");
        }
        if (maxCorretas != null && maxAlternativas != null && maxCorretas >= maxAlternativas) {
            problemas.adicionarProblema(TiposErros.MAX_ALTERNATIVAS, "O nmero mximo de alternativas corretas no pode ser igual ou superior ao nmero mximo de alternativas na questo.");
        }
        if (minCorretas != null && minAlternativas != null && minCorretas >= minAlternativas) {
            problemas.adicionarProblema(TiposErros.MIN_ALTERNATIVAS, "O nmero mnimo de alternativas corretas no pode ser igual ou superior ao nmero mnimo de alternativas na questo.");
        }

        // Em caso de no haver problemas, procura o exame correspondente.
        Exame e = null;
        if (problemas.isVazio()) {
            try {
                e = Exame.pesquisarPorId(idExame);
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }

        // Em caso de no haver problemas, cadastra ou altera a questo.
        Questao q = null;
        if (problemas.isVazio()) {
            try {
                if (idQuestao == null) {
                    q = e.novaQuestao(enunciado, usuario, minCorretas, maxCorretas, minAlternativas, maxAlternativas, alternativasLimpo, corretas);
                } else {
                    q = Questao.pesquisarPorId(idQuestao);
                    q.novaVersao(enunciado, usuario, minCorretas, maxCorretas, minAlternativas, maxAlternativas, alternativasLimpo, corretas);
                }
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }
        questao = q;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    @JsonMarshal
    public String getEnunciado() {
        return enunciado;
    }

    @JsonMarshal
    public Long getIdExame() {
        return idExame;
    }

    @JsonMarshal
    public Long getIdQuestao() {
        return idQuestao;
    }

    @JsonMarshal
    public Set<Integer> getCorretas() {
        return corretas;
    }

    @JsonMarshal
    public List<String> getAlternativas() {
        return alternativas;
    }

    @JsonMarshal
    public Integer getMinCorretas() {
        return minCorretas;
    }

    @JsonMarshal
    public Integer getMaxCorretas() {
        return maxCorretas;
    }

    @JsonMarshal
    public Integer getMinAlternativas() {
        return minAlternativas;
    }

    @JsonMarshal
    public Integer getMaxAlternativas() {
        return maxAlternativas;
    }

    /**
     * Retorna a questo resultante do processo de cadastro.
     * @return O bean de questo resultante do processo de cadastro ou
     * {@code null} em caso de o cadastro no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Questao getQuestao() {
        return questao;
    }

    /**
     * Indica se o processo de cadastro da questo foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    /**
     * Retorna o usurio resultante do processo de cadastro.
     * @return O bean do usurio resultante do processo de cadastro ou
     * {@code null} em caso de o cadastro no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de cadastro de
     * usurio.
     * @return Os problemas resultantes do processo de cadastro.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }
}