/*
 * 13/03/2009, 18:38.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.exame.busca;

import br.com.simuquiz.entidades.Exame;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.util.JsonMarshal;
import java.util.Collection;

/**
 * Classe responsvel por representar a funcionalidade de pesquisa de exames,
 * contendo o texto da busca informado pelo usurio, a resposta a ser enviada e
 * os erros que tiverem ocorrido no processo.
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeBuscarExames {

    private static final long serialVersionUID = -6435910495455155840L;

    /** Os dados de entrada especficos da estratgia da busca. */
    private final Entrada entrada;

    /** ndice do primeiro resultado encontrado. */
    private final int primeiro;

    /** Quantidade de elementos solicitados na pgina. */
    private final int quantidade;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** A lista de beans de exames resultantes da busca. */
    private final Collection<Exame> exames;

    public enum TiposErros { ERRO }

    public static FuncionalidadeBuscarExames buscarPorTexto(String busca, Integer primeiro, Integer quantidade) {
        return new FuncionalidadeBuscarExames(new BuscaExamesStrategy(busca), primeiro, quantidade);
    }

    public static FuncionalidadeBuscarExames buscarPorProprietario(Usuario usuario, Integer primeiro, Integer quantidade) {
        return new FuncionalidadeBuscarExames(new MeusExamesStrategy(usuario), primeiro, quantidade);
    }

    /**
     * Realiza uma busca de exames, que resulta em uma instncia desta classe
     * contendo os dados da busca, uma pgina contendo os exames encontrados
     * como resultado da busca e tambm os problemas ocorridos.
     * @param strategy A estratgia usada na busca.
     * @param primeiro O primeiro resultado da pgina a ser encontrado. Caso
     * seja {@code null}, o valor 0 e assumido.
     * @param quantidade Quantidade de resultados a trazer na pgina. Caso seja
     * {@code null}, o valor 10 e assumido.
     */
    private FuncionalidadeBuscarExames(Strategy strategy, Integer primeiro, Integer quantidade) {

        if (primeiro == null) primeiro = 0;
        if (quantidade == null) quantidade = 10;

        this.entrada = strategy.getEntrada();
        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);
        this.primeiro = primeiro;
        this.quantidade = quantidade;

        if (primeiro < 0) {
            problemas.adicionarProblema(TiposErros.ERRO, "Posio invlida para o primeiro elemento da pesquisa.");
        }
        if (quantidade < 1) {
            problemas.adicionarProblema(TiposErros.ERRO, "Quantidade de resultados invlida para a pesquisa.");
        }

        Collection<Exame> e = null;
        if (problemas.isVazio()) {
            try {
                e = strategy.obterExames(primeiro, quantidade);
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }
        exames = e;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    @JsonMarshal
    public int getPrimeiro() {
        return primeiro;
    }

    @JsonMarshal
    public int getQuantidade() {
        return quantidade;
    }

    /**
     * Indica se o processo de busca bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    @JsonMarshal
    public Entrada getEntrada() {
        return entrada;
    }

    @JsonMarshal
    public Collection<Exame> getExames() {
        return exames;
    }

    @JsonMarshal("temMais")
    public boolean temMais() {
        return exames != null && exames.size() == quantidade;
    }

    /**
     * Retorna a lista de problemas ocorridos na busca.
     * @return Os problemas ocorridos na busca.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }
}
