/*
 * 07/03/2009, 22:19.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.entidades;

import br.com.simuquiz.conexao.Database;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.persistence.Entity;
import javax.persistence.EntityManager;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Query;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Version;
import org.apache.lucene.analysis.StopAnalyzer;
import org.apache.lucene.queryParser.MultiFieldQueryParser;
import org.apache.lucene.queryParser.ParseException;
import org.hibernate.search.annotations.DocumentId;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Index;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.IndexedEmbedded;
import org.hibernate.search.annotations.Store;
import org.hibernate.search.jpa.FullTextEntityManager;
import org.hibernate.search.jpa.Search;

/**
 * Representa um exame cadastrado do sistema.
 * No exame podem ser cadastradas questes e os usurios podem fazer provas com
 * as questes do exame.
 * @author Victor Williams Stafusa da Silva
 */
@Entity
@JsonMarshal
@Indexed
public class Exame implements Serializable {

    private static final long serialVersionUID = -4223109482321536515L;

    /** Chave primria do exame. */
    @Id
    @GeneratedValue
    @DocumentId
    private long idExame;

    /** Nome do exame. */
    @Field(index=Index.TOKENIZED, store=Store.YES)
    private String nome;

    @ManyToOne(fetch=FetchType.LAZY)
    @JoinColumn(name="Id_Usuario_Proprietario")
    private Usuario proprietario;

    //@ManyToMany
    //@JoinTable(name="Moderacao",
    //    joinColumns={@JoinColumn(name="Id_Exame")},
    //    inverseJoinColumns={@JoinColumn(name="Id_Usuario")}
    //)
    //private Collection<Usuario> moderadores;

    /** Decrio curta do exame, para ser mostrada em resultados de buscas. */
    @Field(index=Index.TOKENIZED, store=Store.YES)
    private String descricaoCurta;

    /** Decrio detalhada do exame. */
    @Field(index=Index.TOKENIZED, store=Store.YES)
    private String descricaoCompleta;

    /** Data e hora em que o exame foi criado. */
    @Temporal(TemporalType.TIMESTAMP)
    private Date dataCriacao;

    @IndexedEmbedded
    @OneToMany(fetch=FetchType.LAZY)
    private Collection<Questao> questoes;

    @Version
    private Integer version;

    public Exame() {}

    @JsonMarshal
    public long getIdExame() {
        return idExame;
    }

    @JsonMarshal
    public String getNome() {
        return nome;
    }

    public void setNome(String nome) {
        this.nome = nome;
    }

    @JsonMarshal
    public Usuario getProprietario() {
        return proprietario;
    }

    /*public Collection<Usuario> getModeradores() {
        return moderadores;
    }*/

    @JsonMarshal
    public String getDescricaoCurta() {
        return descricaoCurta;
    }

    public void setDescricaoCurta(String descricaoCurta) {
        this.descricaoCurta = descricaoCurta;
    }

    @JsonMarshal
    public String getDescricaoCompleta() {
        return descricaoCompleta;
    }

    public void setDescricaoCompleta(String descricaoCompleta) {
        this.descricaoCompleta = descricaoCompleta;
    }

    @JsonMarshal
    public Date getDataCriacao() {
        return dataCriacao;
    }

    public void salvar() {
        Database.merge(this);
    }

    public Questao novaQuestao(final String enunciado, final Usuario revisor, final int minCorretas, final int maxCorretas, final int minAlternativas, final int maxAlternativas, final List<String> textoAlternativas, final Set<Integer> corretas) {
        return Database.efetuarTransacao(new Database.Transacao<Questao>() {
            @Override
            public Questao efetuarTransacao(EntityManager em) {
                Questao questao = Questao.novo(em, Exame.this, enunciado, revisor, minCorretas, maxCorretas, minAlternativas, maxAlternativas, textoAlternativas, corretas);
                if (Exame.this.questoes == null) Exame.this.questoes = new ArrayList<Questao>();
                Exame.this.questoes.add(questao);
                em.merge(Exame.this);
                return questao;
            }
        });
    }

    public Collection<Questao> getQuestoes() {
        return questoes;
    }

    public double ranking() {
        if (questoes == null || questoes.size() == 0) return 0;
        int quantidade = questoes.size();
        double soma = 0.0;
        for (Questao q : questoes) {
            soma += q.getVersaoEmUso().getProbabilidadeDeAcertar();
        }
        return soma / quantidade;
    }

    @SuppressWarnings("unchecked")
    public Collection<Questao> pesquisarQuestoes(int primeiro, int quantidade) {
        Query query = Database.getEntityManager().createQuery("select q from Questao q where q.exame = :exame");
        query.setParameter("exame", this);
        query.setFirstResult(primeiro);
        query.setMaxResults(quantidade);
        return (Collection<Questao>) query.getResultList();
    }

    /**
     * Cadastra um novo exame no banco de dados.
     * @param proprietario O proprietrio e criador do exame.
     * @param nome O nome do exame.
     * @param descricaoCurta A descrio curta e resumida do exame.
     * @param descricaoCompleta A descrio completa e detalhada do exame.
     * @return O bean de exame recm-cadastrado.
     */
    static Exame novo(EntityManager em, Usuario proprietario, String nome, String descricaoCurta, String descricaoCompleta) {
        Exame exame = new Exame();
        exame.nome = nome;
        exame.proprietario = proprietario;
        exame.descricaoCurta = descricaoCurta;
        exame.descricaoCompleta = descricaoCompleta;
        exame.dataCriacao = new Date();

        em.persist(exame);
        return exame;
    }

    /**
     * Procura um exame no banco de dados com a chave primria informada.
     * @param idExame A chave primria.
     * @return O bean do exame correspondente, ou {@code null} se no existir
     * exame com a chave informada.
     */
    public static Exame pesquisarPorId(long idExame) {
        return Database.getEntityManager().find(Exame.class, idExame);
    }

    private static final String[] STOP_WORDS = {
        "a","ainda","alem","algum","alguns","ambas","ambos","antes",
        "ao","aonde","aos","apos","aquela","aquelas","aquele","aqueles","aquilo",
        "as","assim","com","como","contra","contudo","cuja","cujas","cujo",
        "cujos","da","daquela","daquelas","daquele","daqueles","daquilo","das",
        "de","dela","delas","dele","deles","demais","depois","desde",
        "dessa","dessas","desse","desses","desta","destas","deste","destes",
        "dispoe","dispoem","disso","disto","diversa", "diversas","diversos",
        "do","dos","durante","e","ela","elas","ele","eles","em","enquanto",
        "entao","entre","essa","essas","esse","esses","esta","estas","este","estes",
        "ha","isso","isto","logo","mais","mas","mediante","menos","menas",
        "mesma","mesmas","mesmo","mesmos","na","nao",
        "naquela","naquelas","naquele","naqueles","naquilo","nas","nem","nenhum","nenhuma",
        "nessa","nessas","nesse","nesses","nesta","nestas","neste","nestes","nisso","nisto",
        "nos","o","onde","os","ou","outra","outras","outro","outros",
        "para","pela","pelas","pelo","pelos","perante","pois","por",
        "porque","portanto","pra","pro","proprio","proprios","propios","quais",
        "quaisquer","qual","qualquer","quando","quanto","que","quem","quer","se",
        "seja","sem","senao","sendo","seu","seus","so","sob","sobre","sua",
        "suas","tal","tambem","tem","tendo","teu","teus","toda","todas","todo",
        "todos","tua","tuas","tudo","um","uma","umas","uns","varias"};

    private static final Map<String, Float> BOOSTS;
    static {
        Map<String, Float> b = new HashMap<String, Float>();
        b.put("nome", 50.0f);
        b.put("descricaoCurta", 20.0f);
        b.put("descricaoCompleta", 10.0f);
        b.put("questoes.versaoEmUso.enunciado", 3.0f);
        b.put("questoes.versaoEmUso.alternativas.texto", 1.0f);
        BOOSTS = Collections.unmodifiableMap(b);
    }

    @SuppressWarnings("unchecked")
    public static Collection<Exame> pesquisarPorDescricao(String texto, int primeiro, int quantidade) {

        FullTextEntityManager fullTextEntityManager = Search.createFullTextEntityManager(Database.getEntityManager());
        MultiFieldQueryParser parser = new MultiFieldQueryParser(new String[]
            {"nome", "descricaoCurta", "descricaoCompleta", "questoes.versaoEmUso.enunciado", "questoes.versaoEmUso.alternativas.texto"},
            new StopAnalyzer(STOP_WORDS), BOOSTS);

        try {
            org.apache.lucene.search.Query luceneQuery = parser.parse(texto);
            Query query = fullTextEntityManager.createFullTextQuery(luceneQuery, Exame.class);
            query.setFirstResult(primeiro);
            query.setMaxResults(quantidade);
            return (Collection<Exame>) query.getResultList();
        } catch (ParseException e) {
            throw new IllegalArgumentException(e);
        }
    }

    @SuppressWarnings("unchecked")
    public static Collection<Exame> pesquisarTudo(int primeiro, int quantidade) {
        Query query = Database.getEntityManager().createQuery("select e from Exame e order by e.nome");
        query.setFirstResult(primeiro);
        query.setMaxResults(quantidade);
        return (Collection<Exame>) query.getResultList();
    }
}
