/*
 * 07/02/2008, 20:04.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */

package br.com.simuquiz.conexao;

import br.com.simuquiz.antispam.Trainer;
import br.com.simuquiz.antispam.classifiers.Classifier;
import br.com.simuquiz.antispam.classifiers.FischerClassifier;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.Persistence;

/**
 * Classe responsvel por fazer a conexo com o banco de dados via JPA.
 * @author Victor Williams Stafusa da Silva
 * @author Pedro Lopes de Souza
 * @author Andr Luiz Pereira lvares
 */
public class Database {

    private Database() {}

    public static Trainer getTrainer() {
        return PopulaBancoDeDados.getTrainer();
    }

    public static Classifier getClassifier() {
        return new FischerClassifier(PopulaBancoDeDados.getTrainer());
    }

    private static EntityManager em;

    private static Map<Thread, EntityManager> HOLD = new ConcurrentHashMap<Thread, EntityManager>();

    private static Thread LIMPADOR = new Thread(new Runnable() {
        @Override
        public void run() {
            while (emf != null && !HOLD.isEmpty()) {
                for (Thread t : HOLD.keySet()) {
                    if (t.isAlive()) continue;
                    EntityManager em = HOLD.get(t);
                    em.close();
                    HOLD.remove(t);
                }
                try {
                    Thread.sleep(600000); // 5 minutos.
                } catch (InterruptedException e) { /* No faz nada. */ }
            }
        }
    });

    /**
     * {@linkplain EntityManagerFactory} utilizado para criar o
     * {@linkplain EntityManager} do projeto.
     */
    private static EntityManagerFactory emf;

    /**
     * Cria e inicializa o mapeamento objeto-relacional do JPA e o
     * {@linkplain EntityManager} do projeto.
     */
    public static void init() {
        emf = Persistence.createEntityManagerFactory("simuquiz");
        PopulaBancoDeDados.popularBancoDeDados();
    }

    /**
     * Obtm o {@linkplain EntityManager} utilizado pelo JPA para a persitnica
     * no banco de dados.
     * @return O {@linkplain EntityManager} do projeto.
     */
    public static EntityManager getEntityManager() {
        /*EntityManager em = HOLD.get(Thread.currentThread());
        if (em != null) return em;
        if (emf == null) throw new IllegalStateException("O banco de dados est sendo finalizado.");
        em = emf.createEntityManager();
        HOLD.put(Thread.currentThread(), em);
        return em;*/
        if (em == null) em = emf.createEntityManager();
        return em;
    }

    /**
     * Fecha o banco de dados para encerrar a aplicao de forma limpa.
     */
    public static void close() {
        if (emf != null) emf.close();
        emf = null;
        LIMPADOR.interrupt();
    }

    public static interface Transacao<T> {
        public T efetuarTransacao(EntityManager em);
    }

    public static <T> T efetuarTransacao(Transacao<T> t) {
        EntityManager em = getEntityManager();
        EntityTransaction et = em.getTransaction();

        T retornado;
        try {
            et.begin();
            retornado = t.efetuarTransacao(em);
            et.commit();
            return retornado;
        } finally {
            if (et.isActive()) et.rollback();
        }
    }

    /**
     * Cria um registro no banco de dados equivalente ao objeto dado.
     * O JPA  responsvel por determinar que tipo de objeto  esse e como
     * persisti-lo.
     * @param objeto O objeto a ser salvo.
     */
    public static void persist(final Object objeto) {
        efetuarTransacao(new Transacao<Void>() {

            @Override
            public Void efetuarTransacao(EntityManager em) {
                em.persist(objeto);
                return null;
            }
        });
    }

    /**
     * Salva um registro existente no banco de dados equivalente ao objeto dado.
     * O JPA  responsvel por determinar que tipo de objeto  esse e como
     * persisti-lo.
     * @param objeto O objeto a ser salvo.
     */
    public static void merge(final Object objeto) {
        efetuarTransacao(new Transacao<Void>() {

            @Override
            public Void efetuarTransacao(EntityManager em) {
                em.merge(objeto);
                return null;
            }
        });
    }

    public static void remove(final Object objeto) {
        efetuarTransacao(new Transacao<Void>() {

            @Override
            public Void efetuarTransacao(EntityManager em) {
                em.remove(objeto);
                return null;
            }
        });
    }
}
