/*
 * 19/05/2009, 09:01.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.testes.web;

import br.com.simuquiz.testes.TesteDoTeste;
import static javax.servlet.http.HttpServletResponse.*;
import br.com.simuquiz.web.CodigoErro;
import br.com.simuquiz.web.FuncionalidadeHttp;
import br.com.simuquiz.web.HttpMethod;
import br.com.simuquiz.web.MapeamentoInconsistenteException;
import br.com.simuquiz.web.ObjetoHttp;
import br.com.simuquiz.web.Param;
import br.com.simuquiz.web.AplicativoRest;
import org.junit.Test;
import static org.junit.Assert.*;
import static br.com.simuquiz.web.ParamSource.*;
import static br.com.simuquiz.testes.web.MocksMapeadores.*;

/**
 * @author Victor Williams Stafusa da Silva
 */
public class MocksWeb {
    public MocksWeb() {}

    /**
     * Este teste tem por objetivo certificar-se que o teste no est errado
     * por deixar alguma das classes mocks serem classes internas no-estticas.
     *
     * Uma vez que o comilador adiciona um parmetro oculto ao construtor da
     * classe interna no-esttica contendo a referncia a classe externa, os
     * testes quebrariam porque a classe WebInterface testada instancia as
     * classes passadas a ele (os mocks) com base na assinatura do construtor.
     */
    @Test
    public void autoTeste1() {
        TesteDoTeste.verificarSeTodosOsMocksSaoEstaticos();
    }

    @Test
    public void autoTeste2() {
        TesteDoTeste.verificarSeNaoEsqueceuAnotacaoTest();
    }

    /**
     * Este teste tem por objetivo certificar-se que a anotao ObjetoHttp no
     * foi esquecida em nenhum dos mocks pois a classe AplicativoRest verifica a
     * presena desta anotao, e no queremos testes falhando ou passando
     * erroneamente por causa disso. A nica exceo a esta regra  a classe
     * MockZoadoSemMapeador que no tem a anotao porque o objetivo dela 
     * testar exatamente este comportamento da classe AplicativoRest.
     */
    @Test
    public void autoTesteVerificaSeNaoEsqueceuAnotacaoObjetoHttp() {
        for (Class<?> c : this.getClass().getDeclaredClasses()) {
            if (c == MockZoadoSemMapeador.class) continue;
            if (!c.isAnnotationPresent(ObjetoHttp.class)) fail(c.getName() + " est sem o mapeador.");
        }
    }

    /**
     * Testa a criao dos mocks funcionais (os que no tem "Zoado" no nome).
     *
     * Seguindo uma metodologia de testes unitrios purista, este teste deveria
     * ser dividido em vrias dezenas de testes feitos um para cada classe mock.
     * O problema  que isso  uma tarefa muito tediosa, repetitiva e
     * principalmente sujeita a erros (poder-se-ia esquecer de testar algum
     * mock). Portanto preferimos usar um teste one-fits-all para evitar isso.
     * Afinal, o que importa  se a barra fica verde ou vermelha no fim, e no
     * quantos mocks ficam verdes ou quantos ficam vermelhos.
     *
     * Acredite, chegamos a criar mais de 50 testes especficos, um para cada
     * mock, e isso definitivamente no compensa. :)
     */
    @Test
    public void testaCriacaoDosMocksFuncionais() {
        Throwable primeiroErroOcorrido = null;
        Class<?> primeiraAFalhar = null;
        int falharam = 0;
        int total = 0;

        for (Class<?> mockClass : this.getClass().getDeclaredClasses()) {
            try {
                if (mockClass.getName().contains("Zoado")) continue;
                total++;
                new AplicativoRest(mockClass);
            } catch (Throwable t) {
                if (falharam == 0) {
                    primeiroErroOcorrido = t;
                    primeiraAFalhar = mockClass;
                }
                falharam++;
            }
        }
        if (falharam != 0) {
            fail(falharam + " classes de " + total + " falharam. A primeira foi " + primeiraAFalhar.getName() + " - " + primeiroErroOcorrido.getMessage());
        }
    }

    /**
     * Testa a criao dos mocks no-funcionais (os que tem "Zoado" no nome).
     * A ideia de criar-se mocks defeituosos pode parecer sem sentido a primeira
     * vista, mas o motivo  que essas classes so utilizadas como Strategies
     * para a classe WebInterface, que faz uma srie de validaes complexas
     * naquilo que ela recebe. O objetivo  testar estas validaes.
     *
     * Seguindo uma metodologia de testes unitrios purista, este teste deveria
     * ser dividido em vrias dezenas de testes feitos um para cada classe mock.
     * O problema  que isso  uma tarefa muito tediosa, repetitiva e
     * principalmente sujeita a erros (poder-se-ia esquecer de testar algum
     * mock). Portanto preferimos usar um teste one-fits-all para evitar isso.
     * Afinal, o que importa  se a barra fica verde ou vermelha no fim, e no
     * quantos mocks ficam verdes ou quantos ficam vermelhos.
     *
     * Acredite, chegamos a criar mais de 50 testes especficos, um para cada
     * mock, e isso definitivamente no compensa. :)
     */
    @Test
    public void testaCriacaoDosMocksZoados() {
        Throwable primeiroErroOcorrido = null;
        Class<?> primeiraAFalhar = null;
        int falharam = 0;
        int total = 0;

        for (Class<?> mockClass : this.getClass().getDeclaredClasses()) {
            if (!mockClass.getName().contains("Zoado")) continue;
            total++;
            try {
                new AplicativoRest(mockClass);
                throw new Exception(mockClass.getName() + " deveria ter dado MapeamentoInconsistenteException.");
            } catch (MapeamentoInconsistenteException esperado) {
            } catch (Throwable t) {
                if (falharam == 0) {
                    primeiroErroOcorrido = t;
                    primeiraAFalhar = mockClass;
                }
                falharam++;
            }
        }
        if (falharam != 0) {
            fail(falharam + " classes de " + total + " falharam. A primeira foi " + primeiraAFalhar.getName() + " - " + primeiroErroOcorrido.getMessage());
        }
    }

    // Mocks com mapeadores defeituosos.

    public static class MockZoadoSemMapeador {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoAbstrato.class)
    public static class MockZoadoComMapeadorAbstrato {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoInterface.class)
    public static class MockZoadoComMapeadorInterface {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoPackagePrivate.class)
    public static class MockZoadoComMapeadorNaoPublico {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoComExcecaoNoConstrutor.class)
    public static class MockZoadoComMapeadorExcecao {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoComConstrutorErrado.class)
    public static class MockZoadoComMapeadorErrado {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorZoadoComConstrutorPrivado.class)
    public static class MockZoadoComMapeadorSemConstrutorPublico {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    // Mocks que no podem ser instanciados corretamente.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemConstrutorPublico {
        private MockZoadoSemConstrutorPublico() {}

        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    static class MockZoadoPackagePrivate {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComConstrutorErrado {
        public MockZoadoComConstrutorErrado(String x) {}

        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComExcecaoNoConstrutor {
        public MockZoadoComExcecaoNoConstrutor() {
            throw new MockException();
        }

        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static interface MockZoadoInterface {
        @FuncionalidadeHttp
        public int foo();
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static abstract class MockZoadoAbstrato {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    // Mocks com problemas na assinatura dos mtodos.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComOParametroNaoAnotado {
        @FuncionalidadeHttp
        public int foo(int x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComVariosParametrosNaoAnotados {
        @FuncionalidadeHttp
        public int foo(int x, String y, long z) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComUmParametroNaoAnotado {
        @FuncionalidadeHttp(caminhos="d/?c")
        public int foo(@Param("a") int x, @Param("b") int y, @Deprecated String z, @Param(value="c", tipo=URL) String a) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComVariosMetodosComUmParametroNaoAnotado {
        @FuncionalidadeHttp
        public int faa(@Param("a") String x, @Param("b") int y) { return 23; }

        @FuncionalidadeHttp
        public int fee(@Param("a") String x, @Param("b") int y) { return 23; }

        @FuncionalidadeHttp
        public int fii(@Deprecated String x, @Param("b") int y) { return 23; }

        @FuncionalidadeHttp
        public int foo(@Param("a") String x, @Param("b") int y) { return 23; }

        @FuncionalidadeHttp
        public int fuu(@Param("a") String x, @Param("b") int y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemMetodos {}

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemMetodosAnotados {
        public String foo() { return "x"; }

        public void bar() {}
    }

    // Mocks com problemas no mtodo HTTP.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemMetodoHttp {
        @FuncionalidadeHttp(tipo={})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComMetodoHttpRepetido {
        @FuncionalidadeHttp(tipo={HttpMethod.DELETE, HttpMethod.DELETE})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComMetodoHttpRepetidoNaoObvio {
        @FuncionalidadeHttp(
            tipo={HttpMethod.POST, HttpMethod.GET, HttpMethod.PUT, HttpMethod.POST, HttpMethod.HEAD}
        )
        public int foo() { return 23; }
    }

    // Mocks que testam ambiguidade no caminho.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeSobrecarga {
        @FuncionalidadeHttp
        public int foo() { return 23; }

        @FuncionalidadeHttp
        public int foo(@Param("a") int x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComSobrecargaDesambiguado {
        @FuncionalidadeHttp
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos="foo2")
        public int foo(@Param("a") int x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhoIgualAoDefaultDeOutroMetodo {
        @FuncionalidadeHttp
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos="foo")
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCaminhosExplicitos {
        @FuncionalidadeHttp(caminhos="foo")
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos="goo")
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCaminhosTrocados {
        @FuncionalidadeHttp(caminhos="goo")
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos="foo")
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhosIguais {
        @FuncionalidadeHttp(caminhos="xoo")
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos="xoo")
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeImplicita {
        @FuncionalidadeHttp(caminhos="goo")
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"xoo", "#"})
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComVariosCaminhos {
        @FuncionalidadeHttp(caminhos={"foo", "hjh", "poiy"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"xoo", "goo", "ga", "gaa"})
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockSimplesComCaminhoHierarquico2Niveis {
        @FuncionalidadeHttp(caminhos={"abc/def"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockSimplesComCaminhoHierarquico3Niveis {
        @FuncionalidadeHttp(caminhos={"abc/def/ghi"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComVariosCaminhosHierarquicos {
        @FuncionalidadeHttp(caminhos={"abc/foo", "def/hjh", "poiy/xoo", "mo"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"xoo/goo", "goo/xoo", "ga", "gaa/gaaa/gagaga"})
        public int goo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComVariosCaminhosHierarquicosParcialmenteSobrepostos {
        @FuncionalidadeHttp(caminhos={"abc/foo", "def/hjh", "poiy/xoo", "mo", "goo/goo"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/goo", "goo/xoo", "ga", "gaa/gaaa/gagaga"})
        public int goo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/hoo", "poiy/xooxoo", "ga/gaga", "gag/gaaa/gagaga"})
        public int hoo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComVariosCaminhosHierarquicosIguais {
        @FuncionalidadeHttp(caminhos={"abc/foo", "def/hjh", "poiy/xoo", "mo", "goo/goo"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/goo", "goo/xoo", "ga", "gaa/gaaa/gagaga"})
        public int goo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/hoo", "poiy/xoo", "ga/gaga", "gag/gaaa/gagaga"})
        public int hoo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhosHierarquicosSobrepostos {
        @FuncionalidadeHttp(caminhos={"abc/foo", "def/hjh", "poiy/xoo", "mo", "goo/goo"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/goo", "goo/xoo", "ga", "gaa/gaaa/gagaga"})
        public int goo() { return 23; }

        @FuncionalidadeHttp(caminhos={"abc/hoo", "poiy/xoo", "ga/gaga", "gag/gaaa/gagaga"})
        public int hoo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhoDuplicadoNoMesmoMetodo {
        @FuncionalidadeHttp(caminhos={"abc/def/ghi", "abc/def/ghi"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhoDuplicadoNoMesmoMetodoNaoObvio {
        @FuncionalidadeHttp(caminhos={"c", "abc/def/ghi", "k", "l/m", "abc/def/ghi", "x/n/s"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCaminhoDuplicadoEmMetodosDiferentes {
        @FuncionalidadeHttp(caminhos={"uu/c", "abc/def/ghi", "uu/k", "uu/l/m", "uu/x/n/s"})
        public int foo() { return 23; }

        @FuncionalidadeHttp(caminhos={"c", "abc/def/ghi", "k", "l/m", "x/n/s"})
        public int goo() { return 23; }
    }

    // Mocks que testam coringas no caminho.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemParametrosComCoringa {
        @FuncionalidadeHttp(caminhos="c/?a")
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockSimplesComCoringa {
        @FuncionalidadeHttp(caminhos="c/?foo")
        public int foo(@Param(tipo=URL, value="foo") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCaminhosMultiplosECoringa {
        @FuncionalidadeHttp(caminhos={"c/?foo", "x/d/?foo/g"})
        public int foo(@Param(tipo=URL, value="foo") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaEmRequest {
        @FuncionalidadeHttp(caminhos="c/?foo")
        public int foo(@Param(tipo=REQUEST, value="foo") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaEmSession {
        @FuncionalidadeHttp(caminhos="x/?foo")
        public int foo(@Param(tipo=SESSION, value="foo") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringasAMais {
        @FuncionalidadeHttp(caminhos="x/?foo/?bar")
        public int foo(@Param(tipo=URL, value="foo") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringasAMenos {
        @FuncionalidadeHttp(caminhos="c/?foo")
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComDoisCoringas {
        @FuncionalidadeHttp(caminhos="c/?foo/?bar")
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringasSeparados {
        @FuncionalidadeHttp(caminhos={"?foo/c/?bar"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringaRepetido {
        @FuncionalidadeHttp(caminhos={"c/?foo/?bar/?foo", "x/?bar/?foo/?bar"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringasNoComeco {
        @FuncionalidadeHttp(caminhos={"?foo/?bar/c"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringasAparentementeAmbiguosAPrimeiraVistaMasQueEstaoOk {
        @FuncionalidadeHttp(caminhos={"?foo/c/?bar"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }

        @FuncionalidadeHttp(caminhos={"?bar/?foo/c"})
        public int goo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringasAmbiguos {
        @FuncionalidadeHttp(caminhos={"?foo/c/?bar"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }

        @FuncionalidadeHttp(caminhos={"?bar/c/?foo"})
        public int goo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="bar") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaSemNome {
        @FuncionalidadeHttp(caminhos={"x/?"})
        public int foo(@Param(tipo=URL, value="xx") String x) { return 23; }
    }


    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComVazioNoCoringa {
        @FuncionalidadeHttp(caminhos={"x/?"})
        public int foo(@Param(tipo=URL, value="") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComEspacoNoCoringa {
        @FuncionalidadeHttp(caminhos={"x/? /"})
        public int foo(@Param(tipo=URL, value=" ") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComEspacoNoNomeCoringa {
        @FuncionalidadeHttp(caminhos={"x/?ab cd/"})
        public int foo(@Param(tipo=URL, value="ab cd") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComEspacoNoComecoDoNomeDoCoringa1 {
        @FuncionalidadeHttp(caminhos={"x/? abcd/"})
        public int foo(@Param(tipo=URL, value="abcd") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComEspacoNoComecoDoNomeDoCoringa2 {
        @FuncionalidadeHttp(caminhos={"x/? abcd/"})
        public int foo(@Param(tipo=URL, value=" abcd") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaComParametroSemNome {
        @FuncionalidadeHttp(caminhos={"x/?foo"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="") String z) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoSemCoringaComParametroURL {
        @FuncionalidadeHttp(caminhos={"x"})
        public int foo(@Param(tipo=URL, value="xx") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaDoisInterrogacao1 {
        @FuncionalidadeHttp(caminhos={"x/??foo/"})
        public int foo(@Param(tipo=URL, value="foo") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaDoisInterrogacao2 {
        @FuncionalidadeHttp(caminhos={"x/??foo/"})
        public int foo(@Param(tipo=URL, value="?foo") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoFim1 {
        @FuncionalidadeHttp(caminhos={"x/foo?/"})
        public int foo(@Param(tipo=URL, value="foo") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoFim2 {
        @FuncionalidadeHttp(caminhos={"x/foo?/"})
        public int foo(@Param(tipo=URL, value="foo?") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoFim3 {
        @FuncionalidadeHttp(caminhos={"x/foo?/"})
        public int foo(@Param(tipo=URL, value="") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoMeio {
        @FuncionalidadeHttp(caminhos={"x/f?oo/"})
        public int foo(@Param(tipo=URL, value="foo") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoComecoEFim1 {
        @FuncionalidadeHttp(caminhos={"x/?foo?/"})
        public int foo(@Param(tipo=URL, value="foo") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComInterrogacaoNoComecoEFim2 {
        @FuncionalidadeHttp(caminhos={"x/?foo?/"})
        public int foo(@Param(tipo=URL, value="foo?") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaSemNomeDoisInterrogacao1 {
        @FuncionalidadeHttp(caminhos={"x/??/"})
        public int foo(@Param(tipo=URL, value="") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaSemNomeDoisInterrogacao2 {
        @FuncionalidadeHttp(caminhos={"x/??/"})
        public int foo(@Param(tipo=URL, value="?") String w) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaDeNomesIguais1 {
        @FuncionalidadeHttp(caminhos={"x/?foo/?foo"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="foo") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComCoringaDeNomesIguais2 {
        @FuncionalidadeHttp(caminhos={"x/?foo/"})
        public int foo(@Param(tipo=URL, value="foo") String x, @Param(tipo=URL, value="foo") String y) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCaractereEspecialNoCoringa {
        @FuncionalidadeHttp(caminhos={"x/?#"})
        public int foo(@Param(tipo=URL, value="#") String x) { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComNumeroNoCoringa {
        @FuncionalidadeHttp(caminhos={"x/?1/"})
        public int foo(@Param(tipo=URL, value="1") String a)
        { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComMuitosCoringas {
        @FuncionalidadeHttp(caminhos={"a/?a/?b/?c/?d/?e/x/?f/?g/?h/?i/?j/g/?k/?l/?m/?n/?o/k"})
        public int foo(@Param(tipo=URL, value="a") String a,
                @Param(tipo=URL, value="b") String b,
                @Param(tipo=URL, value="c") String c,
                @Param(tipo=URL, value="d") String d,
                @Param(tipo=URL, value="e") String e,
                @Param(tipo=URL, value="f") String f,
                @Param(tipo=URL, value="g") String g,
                @Param(tipo=URL, value="h") String h,
                @Param(tipo=URL, value="i") String i,
                @Param(tipo=URL, value="j") String j,
                @Param(tipo=URL, value="k") String k,
                @Param(tipo=URL, value="l") String l,
                @Param(tipo=URL, value="m") String m,
                @Param(tipo=URL, value="n") String n,
                @Param(tipo=URL, value="o") String o)
        { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringasDeTrasParaFrente {
        @FuncionalidadeHttp(caminhos={"x/?a/?b/?c/?d/?e"})
        public int foo(@Param(tipo=URL, value="e") String a,
                @Param(tipo=URL, value="d") String b,
                @Param(tipo=URL, value="c") String c,
                @Param(tipo=URL, value="b") String d,
                @Param(tipo=URL, value="a") String e)
        { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComCoringasForaDeOrdem {
        @FuncionalidadeHttp(caminhos={"x/?a/?b/?c/?d/?e/?f"})
        public int foo(@Param(tipo=URL, value="c") String a,
                @Param(tipo=URL, value="b") String b,
                @Param(tipo=URL, value="f") String c,
                @Param(tipo=URL, value="d") String d,
                @Param(tipo=URL, value="e") String e,
                @Param(tipo=URL, value="a") String f)
        { return 23; }
    }

    // Mocks que testam caminhos mal-formados.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComDuasBarrasNoCaminho {
        @FuncionalidadeHttp(caminhos={"//"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComDuasBarrasInvertidasNoCaminho {
        @FuncionalidadeHttp(caminhos={"\\\\"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComBarrasMisturadas {
        @FuncionalidadeHttp(caminhos={
            "a/b", "a\\c", "/a/d/", "\\a\\e\\", "/a/f", "\\a\\g", "a/h/", "a\\i\\",
            "/x\\a/a", "\\x/y\\b", "/z\\", "\\w/"
        })
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeNasBarras1 {
        @FuncionalidadeHttp(caminhos={"a/b", "a\\b"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeNasBarras2 {
        @FuncionalidadeHttp(caminhos={"a/b", "/a/b"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeNasBarras3 {
        @FuncionalidadeHttp(caminhos={"a/b", "a/b/"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeNasBarras4 {
        @FuncionalidadeHttp(caminhos={"a/b", "/a/b/"})
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockZoadoComAmbiguidadeNasBarras5 {
        @FuncionalidadeHttp(caminhos={"a/b", "\\a\\b\\"})
        public int foo() { return 23; }
    }

    // Mocks que testam compatibilidade dos parmetros. (TO DO)

    // Mocks que testam ordem dos parmetros. (TO DO)

    // Mocks que testam o mapeamento dos cdigos das excees. (TO DO)

    // Mocks bsicos e genricos que no fazem nada, mas funcionam.
    // Se os testes destes falharem, provavelmente os testes de muitos outros
    // que deveriam ser funcionais falharo tambm.

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockTrivial {
        @FuncionalidadeHttp
        public int foo() { return 23; }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComFuncionalidadeComplexa {
        @FuncionalidadeHttp(
            tipo={HttpMethod.POST, HttpMethod.PUT},
            statusSucessoDefault=SC_CREATED,
            statusErroDefault=SC_INTERNAL_SERVER_ERROR,
            caminhos={"/foo", "/bar/foo", "/xxx/foolo"},
            erros={
                @CodigoErro(codigo=SC_NOT_FOUND, erro=NoClassDefFoundError.class),
                @CodigoErro(codigo=SC_BAD_REQUEST, erro=NullPointerException.class)
            }
        )
        public int foo() {
            return 23;
        }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockComParametros {
        @FuncionalidadeHttp
        public int foo(@Param(value="abc", tipo=SESSION) String abc, @Param("xyz") int a) {
            return 23;
        }
    }

    @ObjetoHttp(MapeadorNaoFazNada.class)
    public static class MockBastanteComplexo {
        @FuncionalidadeHttp(
            tipo={HttpMethod.HEAD, HttpMethod.GET, HttpMethod.POST},
            statusSucessoDefault=SC_OK,
            statusErroDefault=SC_FORBIDDEN,
            caminhos={"/foo", "/bar/foo", "/xxx/foolo"},
            erros={
                @CodigoErro(codigo=SC_NOT_FOUND, erro=NoClassDefFoundError.class),
                @CodigoErro(codigo=SC_INTERNAL_SERVER_ERROR, erro=NullPointerException.class)
            }
        )
        public int foo(@Param(value="abc", tipo=SESSION) String abc, @Param("xyz") int a) {
            return 23;
        }

        @FuncionalidadeHttp(
            tipo={HttpMethod.DELETE, HttpMethod.POST},
            statusSucessoDefault=SC_GONE,
            caminhos={"/goo/?ak/?xxx", "/bar/goo/?xxx/?ak", "/xxx/goolo/?ak/?xxx"},
            erros={
                @CodigoErro(codigo=SC_NOT_FOUND, erro=NullPointerException.class),
                @CodigoErro(codigo=SC_SERVICE_UNAVAILABLE, erro=ArithmeticException.class)
            }
        )
        public String goo(@Param("abc") String abc, @Param("xyz") int a, @Param(value="ak", tipo=URL) String b, @Param(value="xxx", tipo=URL) String x) {
            return "festa no ape";
        }
    }
}
