/*
 * 17/05/2009, 14:16.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.testes.util;

import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.testes.TesteDoTeste;
import java.util.List;
import static org.junit.Assert.*;
import org.junit.Test;

/**
 * @author Victor Williams Stafusa da Silva
 */
public class TesteRelatorioProblemas {

    @Test
    public void autoTeste() {
        TesteDoTeste.verificarSeNaoEsqueceuAnotacaoTest();
        TesteDoTeste.verificarSeTodosOsMocksSaoEstaticos();
    }

    public enum Mock1 { A, B, C }
    public enum Mock2 { X, Y, Z }

    private RelatorioProblemas<Mock1> mock1() {
        return new RelatorioProblemas<Mock1>(Mock1.class);
    }

    @Test
    public void testaInstanciacao() {
        mock1();
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaInstanciacaoComNull() {
        new RelatorioProblemas<Mock1>(null);
    }

    @Test(expected=IllegalArgumentException.class)
    @SuppressWarnings("unchecked")
    public void testaInstanciacaoComClasseSubvertida() {
        new RelatorioProblemas<Mock1>((Class) String.class);
    }

    @Test
    public void testaTipoDoProblemas() {
        assertSame(Mock1.class, mock1().getTipo());
    }

    @Test
    public void testaAdicionarUmProblema() {
        mock1().adicionarProblema(Mock1.A, "foo");
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarUmProblemaNull() {
        mock1().adicionarProblema(null, "foo");
    }

    @Test(expected=ClassCastException.class)
    @SuppressWarnings("unchecked")
    public void testaAdicionarUmProblemaInvalido() {
        ((RelatorioProblemas) mock1()).adicionarProblema(Mock2.X, "foo");
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarUmProblemaComDescricaoNull() {
        mock1().adicionarProblema(Mock1.A, (String) null);
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarUmProblemaComDescricaoVazia() {
        mock1().adicionarProblema(Mock1.A, "");
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarTipoNullComDescricaoNull() {
        mock1().adicionarProblema(null, (String) null);
    }

    @Test
    public void testaAdicionarUmErro() {
        mock1().adicionarProblema(Mock1.A, new Exception("foo"));
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarUmErroComTipoNull() {
        mock1().adicionarProblema(null, new Exception("foo"));
    }

    @Test(expected=ClassCastException.class)
    @SuppressWarnings("unchecked")
    public void testaAdicionarUmErroInvalido() {
        ((RelatorioProblemas) mock1()).adicionarProblema(Mock2.X, new Exception("foo"));
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarUmErroNull() {
        mock1().adicionarProblema(Mock1.A, (Throwable) null);
    }

    @Test
    public void testaAdicionarUmErroSemDescricao() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, new ArithmeticException());
        assertEquals(ArithmeticException.class.getName(), rp.getProblemas(Mock1.A).get(0));
    }

    @Test
    public void testaAdicionarUmErroComDescricaoVazia() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, new ArithmeticException(""));
        assertEquals(ArithmeticException.class.getName(), rp.getProblemas(Mock1.A).get(0));
    }

    @Test(expected=IllegalArgumentException.class)
    public void testaAdicionarTipoNullComErroNull() {
        mock1().adicionarProblema(null, (Throwable) null);
    }

    @Test
    public void testaSeInicialmenteVazio() {
        assertTrue(mock1().isVazio());
    }

    @Test
    public void testaSeNaoVazioAposAdicionar() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        assertFalse(rp.isVazio());
    }

    @Test
    public void testaSeNaoVazioAposAdicionar2Vezes() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        assertFalse(rp.isVazio());
    }

    @Test
    public void testaSeInicialmenteAberto() {
        assertFalse(mock1().isFechado());
    }

    @Test
    public void testaSeAbertoAposAdicionar() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        assertFalse(rp.isFechado());
    }

    @Test
    public void testaSeAbertoAposAdicionar2Vezes() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        assertFalse(rp.isFechado());
    }

    @Test
    public void testaSeFecha() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        rp.fechar();
        assertTrue(rp.isFechado());
    }

    @Test
    public void testaNaoVazioQuandoFechadoNaoVazio() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        rp.fechar();
        assertFalse(rp.isVazio());
    }

    @Test
    public void testaVazioQuandoFechadoVazio() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.fechar();
        assertTrue(rp.isVazio());
    }

    @Test
    public void testaFechadoQuandoFechadoVazio() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.fechar();
        assertTrue(rp.isFechado());
    }

    @Test(expected=IllegalStateException.class)
    public void testaAdicionarProblemaAoFechado() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        rp.fechar();
        rp.adicionarProblema(Mock1.C, "muaaa");
    }

    @Test(expected=IllegalStateException.class)
    public void testaAdicionarErroAoFechado() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "foo");
        rp.adicionarProblema(Mock1.B, "xoo");
        rp.fechar();
        rp.adicionarProblema(Mock1.C, new Exception("muaaa"));
    }

    @Test
    public void testaListarProblemasVazio() {
        assertTrue(mock1().getProblemas(Mock1.A).isEmpty());
    }

    @Test
    public void testaListarProblemasComUmProblema() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "aaa");
        assertEquals(1, rp.getProblemas(Mock1.A).size());
        assertTrue(rp.getProblemas(Mock1.A).contains("aaa"));
    }

    @Test
    public void testaListarProblemasErrado() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "aaa");
        assertTrue(rp.getProblemas(Mock1.B).isEmpty());
        assertTrue(rp.getProblemas(Mock1.C).isEmpty());
    }

    @Test
    public void testaListarProblemasComDoisProblemas() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "aaa");
        rp.adicionarProblema(Mock1.A, "bbb");
        assertEquals(2, rp.getProblemas(Mock1.A).size());
        assertTrue(rp.getProblemas(Mock1.A).contains("aaa"));
        assertTrue(rp.getProblemas(Mock1.A).contains("bbb"));
        assertTrue(rp.getProblemas(Mock1.B).isEmpty());
        assertTrue(rp.getProblemas(Mock1.C).isEmpty());
    }

    @Test
    public void testaListarProblemasDiferentes() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "aaa");
        rp.adicionarProblema(Mock1.A, "bbb");
        rp.adicionarProblema(Mock1.B, "ccc");
        rp.adicionarProblema(Mock1.B, "ddd");
        rp.adicionarProblema(Mock1.C, "eee");
        rp.adicionarProblema(Mock1.C, "fff");
        rp.adicionarProblema(Mock1.C, "ggg");
        assertEquals(2, rp.getProblemas(Mock1.A).size());
        assertEquals(2, rp.getProblemas(Mock1.B).size());
        assertEquals(3, rp.getProblemas(Mock1.C).size());
        assertTrue(rp.getProblemas(Mock1.A).contains("aaa"));
        assertTrue(rp.getProblemas(Mock1.A).contains("bbb"));
        assertTrue(rp.getProblemas(Mock1.B).contains("ccc"));
        assertTrue(rp.getProblemas(Mock1.B).contains("ddd"));
        assertTrue(rp.getProblemas(Mock1.C).contains("eee"));
        assertTrue(rp.getProblemas(Mock1.C).contains("fff"));
        assertTrue(rp.getProblemas(Mock1.C).contains("ggg"));
    }

    @Test(expected=UnsupportedOperationException.class)
    public void testaListaProblemasImutavel() {
        RelatorioProblemas<Mock1> rp = mock1();
        rp.adicionarProblema(Mock1.A, "aaa");
        List<String> lista = rp.getProblemas(Mock1.A);
        lista.add("Mwhahaha");
    }

    @Test
    public void testaToString() {
        fail("Teste no implementado!");
    }
}
