/*
 * 08/03/2009, 21:46.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.testes.util;

import br.com.simuquiz.testes.TesteDoTeste;
import java.lang.reflect.Array;
import java.lang.reflect.ParameterizedType;
import static br.com.simuquiz.util.ReflectionUtils.*;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import static org.junit.Assert.*;
import org.junit.Test;

/**
 * @author Victor Williams Stafusa da Silva
 */
public class TesteReflection {

    @Test
    public void autoTeste() {
        TesteDoTeste.verificarSeNaoEsqueceuAnotacaoTest();
    }

    private static final Class<?>[] CLASSES_NON_SENSE = {
        Object.class, TesteReflection.class, Test.class, System.class, ClassLoader.class,
        ArrayList.class, LinkedList.class, Map.class, HashMap.class, Throwable.class,
        Object[].class, String[][].class, int[][].class, System[].class, TypeMock[].class,
        boolean.class, Boolean.class, void.class, Void.class
    };

    private Class<?> array(Class<?> c) {
        return Array.newInstance(c, 0).getClass();
    }

    @Test
    public void testaSeClassesUrlSaoClassesRequest() {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            assertTrue(tipoRequest(c));
        }
    }

    @Test
    public void testaSeClassesNaoUrlNaoSaoClassesRequest() {
        for (Class<?> c : CLASSES_NON_SENSE) {
            assertFalse(tipoRequest(c));
        }
        for (Class<?> c : COLECOES_POSSIVEIS) {
            assertFalse(tipoRequest(c));
        }
    }

    @Test
    public void testaSeArraysDeClassesUrlSaoClassesRequest() {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            assertTrue(tipoRequest(array(c)));
        }
    }

    @Test
    public void testaSeArraysDeClassesNaoUrlNaoSaoClassesRequest() {
        for (Class<?> c : CLASSES_NON_SENSE) {
            // Pula os voids porque no existe array de void.
            if (c == void.class || c == Void.class) continue;

            assertFalse(tipoRequest(array(c)));
        }
        for (Class<?> c : COLECOES_POSSIVEIS) {
            assertFalse(tipoRequest(array(c)));
        }
    }

    public static class TypeMock implements ParameterizedType {

        private final Class<?> cit;
        private final Class<?>[] c;

        public TypeMock(Class<?> cit, Class<?> c) {
            this.c = new Class<?>[] {c};
            this.cit = cit;
        }

        @Override
        public Type[] getActualTypeArguments() {
            return c;
        }

        @Override
        public Type getRawType() {
            return cit;
        }

        @Override
        public Type getOwnerType() {
            return null;
        }
    }

    @Test
    public void testaSeTiposCompostosDeClassesUrlSaoClassesRequest() {
        for (Class<? extends Iterable> cit : COLECOES_POSSIVEIS) {
            for (Class<?> c : TIPOS_POSSIVEIS) {
                assertTrue(tipoRequest(new TypeMock(cit, c)));
            }
        }
    }

    @Test
    public void testaSeTiposCompostosDeClassesNaoUrlNaoSaoClassesRequest() {
        for (Class<? extends Iterable> cit : COLECOES_POSSIVEIS) {
            for (Class<?> c : CLASSES_NON_SENSE) {
                assertFalse(tipoRequest(new TypeMock(cit, c)));
            }
            for (Class<?> c : COLECOES_POSSIVEIS) {
                assertFalse(tipoRequest(new TypeMock(cit, c)));
            }
        }
    }

    @Test
    public void testaSeTiposCompostosInvalidosNaoSaoClassesRequest() {
        for (Class<?> cit : CLASSES_NON_SENSE) {
            for (Class<?> c : CLASSES_NON_SENSE) {
                assertFalse(tipoRequest(new TypeMock(cit, c)));
            }
            for (Class<?> c : COLECOES_POSSIVEIS) {
                assertFalse(tipoRequest(new TypeMock(cit, c)));
            }
            for (Class<?> c : TIPOS_POSSIVEIS) {
                assertFalse(tipoRequest(new TypeMock(cit, c)));
            }
        }
    }

    @Test
    public void representarURLComoNull() throws StringParseException {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            assertNull(representarComoURL(c, null));
        }
    }

    @Test
    public void representarLixoNaURLComoNull() throws StringParseException {
        for (Class<?> c : CLASSES_NON_SENSE) {
            try {
                representarComoURL(c, null);
                fail();
            } catch (IllegalArgumentException esperado) {}
        }

        for (Class<?> c : COLECOES_POSSIVEIS) {
            try {
                representarComoURL(c, null);
                fail();
            } catch (IllegalArgumentException esperado) {}
        }
    }

    @Test
    public void testaRepresentarComoNumero() throws StringParseException {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            if (c == String.class || c == char.class || c == Character.class) continue;
            Object r = representarComoURL(c, "57");
            assertEquals(57, ((Number) r).intValue());
        }
    }

    @Test
    public void representarNumeroComoLixo() throws StringParseException {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            if (c == String.class || c == char.class || c == Character.class) continue;
            try {
                representarComoURL(c, "abc");
                fail();
            } catch (StringParseException esperado) {}
        }
    }

    @Test
    public void testaRepresentarComoNumeroNegativo() throws StringParseException {
        for (Class<?> c : TIPOS_POSSIVEIS) {
            if (c == String.class || c == char.class || c == Character.class) continue;
            Object r = representarComoURL(c, "-57");
            assertEquals(-57, ((Number) r).intValue());
        }
    }

    @Test(expected=StringParseException.class)
    public void testaRepresentarInteiroComoNumeroFracionario() throws StringParseException {
        representarComoURL(Integer.class, "57.5");
    }

    @Test(expected=StringParseException.class)
    public void testaRepresentarInteiroNegativoComoNumeroFracionario() throws StringParseException {
        representarComoURL(Integer.class, "-57.5");
    }

    @Test
    public void testaRepresentarDoubleComoNumeroFracionario() throws StringParseException {
        Object r = representarComoURL(Double.class, "57.5");
        assertEquals(57.5, (Double) r, 0.0001);
    }

    @Test
    public void testaRepresentarDoubleNegativoComoNumeroFracionario() throws StringParseException {
        Object r = representarComoURL(Double.class, "-57.5");
        assertEquals(-57.5, (Double) r, 0.0001);
    }

    @Test
    public void testaRepresentarBigDecimalComoNumeroFracionario() throws StringParseException {
        Object r = representarComoURL(BigDecimal.class, "57.5");
        assertEquals("57.5", ((BigDecimal) r).toPlainString());
    }

    @Test
    public void testaRepresentarBigDecimalNegativoComoNumeroFracionario() throws StringParseException {
        Object r = representarComoURL(BigDecimal.class, "-57.5");
        assertEquals("-57.5", ((BigDecimal) r).toPlainString());
    }
}
