/*
 * 20/05/2009, 06:55.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.web;

import br.com.simuquiz.util.ReflectionUtils;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Victor Williams Stafusa da Silva
 */
public final class FuncionalidadeRest {

    private final Method metodo;
    private final FuncionalidadeHttp func;
    private final String nomeMetodo;
    private final List<CodigoErro> codigos;
    private final Set<Path> paths;
    private final Set<HttpMethod> metodosHttp;
    private final Map<ParamSource, Map<String, Type>> parametrosNaoModificaveis;
    private final int coringas;
    private final List<Param> parametrosOrdenados;
    private final Mapeador mapeador;

    public FuncionalidadeRest(RegistroMapeador mapeadores, Method metodo) {
        this.metodo = metodo;
        this.parametrosOrdenados =
                new ArrayList<Param>(metodo.getGenericParameterTypes().length);
        metodo.setAccessible(true);
        this.func = metodo.getAnnotation(FuncionalidadeHttp.class);
        this.nomeMetodo = metodo.getName();
        this.codigos = validarCodigosErrosHttp();

        ObjetoHttp oh = metodo.getAnnotation(ObjetoHttp.class);
        this.mapeador = (oh == null
                ? mapeadores.getMapeadorDefault() : mapeadores.obterMapeador(oh.value()));

        Map<ParamSource, Map<String, Type>> modf =
                new EnumMap<ParamSource, Map<String, Type>>(ParamSource.class);
        Map<ParamSource, Map<String, Type>> unmodf =
                new EnumMap<ParamSource, Map<String, Type>>(ParamSource.class);

        for (ParamSource ps : ParamSource.values()) {
            Map<String, Type> params = new HashMap<String, Type>();
            modf.put(ps, params);
            unmodf.put(ps, Collections.unmodifiableMap(params));
        }

        parametrosNaoModificaveis = Collections.unmodifiableMap(unmodf);

        this.coringas = validarParametros(modf);
        this.paths = caminho();
        this.metodosHttp = metodosHttp();
    }

    private List<CodigoErro> validarCodigosErrosHttp() {
        CodigoErro[] erros = func.erros();

        // Verifica se a ordem declarada dos erros faz sentido.
        for (int i = 0; i < erros.length - 1; i++) {
            for (int j = i + 1; j < erros.length; j++) {
                if (erros[i].erro() == erros[j].erro()) {
                    throw new MapeamentoInconsistenteException("O mtodo " + nomeMetodo
                            + " declara cdigos de erros para a exceo "
                            + erros[i].erro().getName() + " mais de uma vez.");
                }
                if (erros[i].erro().isAssignableFrom(erros[j].erro())) {
                    throw new MapeamentoInconsistenteException("O mtodo " + nomeMetodo
                            + " no declara os erros na ordem certa. "
                            + erros[i].erro().getName() + "  superclasse de "
                            + erros[j].erro().getName() + " e por isso deveria vir depois.");
                }
            }
        }

        return Arrays.asList(erros);
    }

    private int validarParametros(Map<ParamSource, Map<String, Type>> parametrosModificaveis) {
        int indiceParam = -1;
        int contaCoringas = 0;

        // Itera por cada parmetro.
        e: for (Annotation[] paramAnnots : metodo.getParameterAnnotations()) {
            // Aumenta o contador de parmetros.
            // Na primeira iterao ele deve valer 0.
            indiceParam++;

            // Itera as anotaes do parmetro procurando pela @Param.
            for (Annotation a : paramAnnots) {
                // Pula as anotaes que no forem interessantes.
                if (a.annotationType() != Param.class) continue;

                Param param = ((Param) a);
                ParamSource source = param.tipo();

                if (source == ParamSource.URL) {
                    contaCoringas++;
                }

                // Verifica se o parmetro no  repetido e se  compatvel.
                Type tipo = metodo.getGenericParameterTypes()[indiceParam];
                Map<String, Type> params = parametrosModificaveis.get(source);
                if (params.get(param.value()) != null) {
                    throw new MapeamentoInconsistenteException(
                            "Existe mais de um parmetro do tipo " + source.name()
                            + " com o nome " + param.value() + " no mtodo "
                            + nomeMetodo + ".");
                }
                validarTipoParametro(indiceParam, nomeMetodo, tipo, source);
                params.put(param.value(), tipo);
                parametrosOrdenados.add(param);

                continue e;
            }

            // No achou a anotao @Param. Isso  um erro.
            throw new MapeamentoInconsistenteException("O mtodo "
                    + metodo.getName() + " no tem no " + (indiceParam + 1)
                    + "o parmetro a anotao @Param, que  obrigatria.");
        }
        return contaCoringas;
    }

    private static Path criarPath(String path) {
        try {
            return new Path(path);
        } catch (IllegalArgumentException e) {
            throw new MapeamentoInconsistenteException(e);
        }
    }

    private Set<Path> caminho() {
        String[] cam = func.caminhos();
        Set<Path> caminhos = new HashSet<Path>(cam.length);

        for (String c : cam) {
            Path p = criarPath("#".equals(c) ? nomeMetodo : c);
            if (p.coringas() != coringas) {
                throw new MapeamentoInconsistenteException("O mtodo "
                        + nomeMetodo + (
                            coringas == 0 ? " no especifica parmetros" :
                            coringas == 1 ? " especifica 1 parmetro" :
                            " especifica " + coringas + " parmetros"
                        ) + " de URL, mas o caminho \"" + c
                        + "\" especificado na anotao @FuncionalidadeHttp " + (
                            p.coringas() == 0 ? " no tem parmetros de URL." :
                            p.coringas() == 1 ? " tem 1 parmetro de URL." :
                            " tem " + p.coringas() + " parmetros de URL."));
            }

            if (!caminhos.add(p)) {
                throw new MapeamentoInconsistenteException("O path \"" + c
                        + "\" aparece mais de uma vez na anotao @FuncionalidadeHttp do mtodo "
                        + nomeMetodo + ".");
            }
        }
        return Collections.unmodifiableSet(caminhos);
    }

    private Set<HttpMethod> metodosHttp() {

        HttpMethod[] mets = func.tipo();

        // O mtodo deve responder a pelo menos um dos mtodos HTTP.
        if (mets.length == 0) {
            throw new MapeamentoInconsistenteException("O mtodo " + nomeMetodo
                    + " no responde a nenhum tipo de requisio HTTP.");
        }

        // Guarda os elementos em um Set e verifica se no h repetidos.
        Set<HttpMethod> http = EnumSet.noneOf(HttpMethod.class);
        for (HttpMethod m : mets) {
            if (!http.add(m)) {
                throw new MapeamentoInconsistenteException("O mtodo " + nomeMetodo
                        + " tem o tipo de requisio " + m.name() + " duplicado.");
            }
        }

        return Collections.unmodifiableSet(http);
    }

    public String[] caminhos() {
        return func.caminhos();
    }

    public int coringas() {
        return coringas;
    }

    public Set<Path> getPaths() {
        return paths;
    }

    public Set<HttpMethod> getMetodosHttp() {
        return metodosHttp;
    }

    public String getNome() {
        return nomeMetodo;
    }

    private void validarTipoParametro(int indice, String nome, Type parametro, ParamSource p) {
        switch (p) {
            case SESSION:
                return;

            case REQUEST:
                if (ReflectionUtils.tipoRequest(parametro)) return;

                throw new MapeamentoInconsistenteException("O " + indice
                        + "o parmetro do mtodo " + nome
                        + " tem um tipo que no pode extrado de uma requiso HTTP.");

            case URL:
                if (ReflectionUtils.tipoURL(parametro)) return;

                throw new MapeamentoInconsistenteException("O " + indice
                        + "o parmetro do mtodo " + nome
                        + " tem um tipo que no pode extrado de uma requiso URL.");
        }
    }

    public int codigoErro(Throwable t) {
        for (CodigoErro c : codigos) {
            if (c.erro().isInstance(t)) return c.codigo();
        }
        return func.statusErroDefault();
    }

    public int statusSucessoDefault() {
        return func.statusSucessoDefault();
    }

    public int statusErroDefault() {
        return func.statusErroDefault();
    }

    public Map<ParamSource, Map<String, Type>> tiposParametros() {
        return parametrosNaoModificaveis;
    }

    public Object invocar(Object instancia, Map<ParamSource, Map<String, Object>> parametros) throws InvocationTargetException {
        List<Object> parametrosReais = new ArrayList<Object>(parametrosOrdenados.size());
        for (Param param : parametrosOrdenados) {
            parametrosReais.add(parametros.get(param.tipo()).get(param.value()));
        }

        try {
            return metodo.invoke(instancia, parametrosReais.toArray());
        } catch (IllegalAccessException ex) {
            throw new AssertionError(); // Nunca deveria ocorrer.
        }
    }

    public Object invocar(ParametrosRequisicao parametros) throws InvocationTargetException {
        return invocar(parametros.getInstancia(), parametros.getParametrosReais());
    }

    public void executar(ParametrosRequisicao parametros) throws IOException {
        this.mapeador.executar(parametros);
    }

    @Override
    public String toString() {
        return nomeMetodo;
    }

    public void mapearSe(RegistroCaminho<FuncionalidadeRest> registro) {
        for (HttpMethod t : getMetodosHttp()) {
            for (Path path : getPaths()) {
                List<String> lista = new ArrayList<String>(path.elementos().size() + 1);
                lista.add(t.name());
                lista.addAll(path.elementos());

                if (!registro.put(this, lista)) {
                    throw new MapeamentoInconsistenteException("O path \"" + this.toString()
                            + "\" foi mapeado mais de uma vez. Est presente nos mtodos "
                            + nomeMetodo + " e " + registro.get(lista).nomeMetodo + ".");
                }
            }
        }
    }
}
