/*
 * 18/05/2009, 23:24.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.web;

import br.com.simuquiz.util.ReflectionUtils.StringParseException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * @author Victor Williams Stafusa da Silva
 */
public class AplicativoRest {

    private final Object web;
    private final RegistroCaminho<FuncionalidadeRest> caminhos;
    private final RegistroMapeador mapeadores;
    private final List<FuncionalidadeRest> funcionalidades;

    public AplicativoRest(Class<?> classeWeb) {
        if (classeWeb == null) throw new IllegalArgumentException();

        ObjetoHttp objh = classeWeb.getAnnotation(ObjetoHttp.class);
        if (objh == null) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada porque no est anotada com @ObjetoHttp.");
        }

        try {
            this.web = classeWeb.getConstructor().newInstance();
        } catch (IllegalAccessException e) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada porque o construtor sem argumentos no  pblico.");
        } catch (NoSuchMethodException e) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada pois no tem um construtor sem argumentos.");
        } catch (InstantiationException e) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada porque ela no  uma classe concreta.");
        } catch (InvocationTargetException e) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada porque o construtor lanou uma exceo.", e.getCause());
        }
        this.caminhos = new RegistroCaminho<FuncionalidadeRest>();
        this.mapeadores = new RegistroMapeador(objh.value());
        this.funcionalidades = new ArrayList<FuncionalidadeRest>();

        for (Class<?> c = web.getClass(); c != null; c = c.getSuperclass()) {
            for (Method m : c.getDeclaredMethods()) {
                FuncionalidadeHttp func = m.getAnnotation(FuncionalidadeHttp.class);
                if (func == null) continue;
                FuncionalidadeRest webf = new FuncionalidadeRest(mapeadores, m);
                webf.mapearSe(caminhos);
                funcionalidades.add(webf);
            }
        }

        if (caminhos.isEmpty()) {
            throw new MapeamentoInconsistenteException("A classe " + classeWeb.getName()
                    + " no pode ser utilizada porque no contm nenhum mtodo anotado com @FuncionalidadeHttp.");
        }
        //caminhos.dump();
    }

    public FuncionalidadeRest localizarFuncionalidade(HttpMethod metodo, Path solicitado) {
        List<String> lista = new ArrayList<String>(solicitado.elementos().size() + 1);
        lista.add(metodo.name());
        lista.addAll(solicitado.elementos());
        return caminhos.get(lista);
    }

    private Map<String, String> coringas(HttpMethod metodo, Path solicitado) {
        List<String> lista = new ArrayList<String>(solicitado.elementos().size() + 1);
        lista.add(metodo.name());
        lista.addAll(solicitado.elementos());
        return caminhos.coringas(lista);
    }

    public boolean permite(HttpMethod metodo) {
        return caminhos.permite(metodo);
    }

    public void executar(HttpMethod metodo, HttpServletRequest request, HttpServletResponse response) throws IOException {
        try {
            executarInterno(metodo, request, response);
        } catch (Throwable e) {
            response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, e.toString());
            e.printStackTrace();
            return;
        }
    }

    private void executarInterno(HttpMethod metodo, HttpServletRequest request, HttpServletResponse response) throws IOException {

        // D um 405 se o mtodo no for aceito.
        if (!caminhos.permite(metodo)) {
            response.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED, "O mtodo " + metodo + " no  suportado.");
            return;
        }

        // Descobre qual  o caminho para o recurso.
        // D um 404 se for um caminho mal-formado.
        Path solicitado;
        try {
            solicitado = new Path(request.getPathInfo());
        } catch (IllegalArgumentException ex) {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, "No h nada no caminho \"" + request.getPathInfo() + "\". Alis, nem caminho isso .");
            return;
        }

        // Descobre qual  a funcionalidade mapeada para o caminho do recurso.
        FuncionalidadeRest m = localizarFuncionalidade(metodo, solicitado);

        // Se no achou a funcionalidade, d um 404.
        if (m == null) {
            response.sendError(HttpServletResponse.SC_NOT_FOUND, "No h nada no caminho \"" + solicitado + "\". Isso no existe.");
            return;
        }

        // Delega para a funcionalidade.
        ParametrosRequisicao pr;
        try {
            Map<String, String> coringas = coringas(metodo, solicitado);
            if (coringas == null) throw new AssertionError();
            pr = new ParametrosRequisicao(web, m, request, response, coringas);
        } catch (StringParseException e) {
            response.sendError(HttpServletResponse.SC_BAD_REQUEST, e.toString());
            return;
        }

        m.executar(pr);
    }
}
