/*
 * 19/05/2009, 19:43.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.util;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * @author Victor Williams Stafusa da Silva
 */
public class ReflectionUtils {

    private ReflectionUtils() {}

    public static final Map<Class<?>, Class<?>> EMPACOTADORAS;
    public static final Map<Class<?>, Class<?>> EMPACOTADAS;
    static {
        Map<Class<?>, Class<?>> empacotadoras = new HashMap<Class<?>, Class<?>>(9);
        Map<Class<?>, Class<?>> empacotadas = new HashMap<Class<?>, Class<?>>(9);
        empacotadoras.put(int.class, Integer.class); empacotadas.put(Integer.class, int.class);
        empacotadoras.put(char.class, Character.class); empacotadas.put(Character.class, char.class);
        empacotadoras.put(long.class, Long.class); empacotadas.put(Long.class, long.class);
        empacotadoras.put(float.class, Float.class); empacotadas.put(Float.class, float.class);
        empacotadoras.put(double.class, Double.class); empacotadas.put(Double.class, double.class);
        empacotadoras.put(byte.class, Byte.class); empacotadas.put(Byte.class, byte.class);
        empacotadoras.put(short.class, Short.class); empacotadas.put(Short.class, short.class);
        empacotadoras.put(boolean.class, Boolean.class); empacotadas.put(Boolean.class, boolean.class);
        empacotadoras.put(void.class, Void.class); empacotadas.put(Void.class, void.class);
        EMPACOTADORAS = Collections.unmodifiableMap(empacotadoras);
        EMPACOTADAS = Collections.unmodifiableMap(empacotadas);
    }

    /**
     * Tipos de dados possveis de serem representadas em uma URL.
     */
    public static final Set<Class<?>> TIPOS_POSSIVEIS;

    static {
        Set<Class<?>> tipos = new HashSet<Class<?>>(17);
        tipos.addAll(Arrays.asList(
            new Class<?>[] {
                byte.class, Byte.class, short.class, Short.class, char.class,
                Character.class, int.class, Integer.class, long.class,
                Long.class, float.class, Float.class, double.class, Double.class,
                BigInteger.class, BigDecimal.class, String.class
            }));
        TIPOS_POSSIVEIS = Collections.unmodifiableSet(tipos);
    }

    /**
     * Determina se um tipo de dados pode ser representado em uma URL.
     * <br />
     * Os tipos de dados possveis de serem representados em uma URL so
     * {@linkplain String}s e nmeros.
     * @param tipo O tipo de dados a ser determinado se pode ser representado em
     * uma URL.
     * @return Se  possvel representar.
     */
    public static boolean tipoURL(Type tipo) {
        return TIPOS_POSSIVEIS.contains(tipo);
    }

    /**
     * Tipos de colees possveis de serem representadas em uma requisio
     * HTTP. So as interfaces {@linkplain Iterable}, {@linkplain Collection},
     * {@linkplain List}, {@linkplain Set}, {@linkplain SortedSet}.
     */
    public static final Set<Class<? extends Iterable>> COLECOES_POSSIVEIS;

    static {
        Set<Class<? extends Iterable>> tipos =
                new HashSet<Class<? extends Iterable>>(5);

        tipos.add(Iterable.class);
        tipos.add(Collection.class);
        tipos.add(List.class);
        tipos.add(Set.class);
        tipos.add(SortedSet.class);
        COLECOES_POSSIVEIS = Collections.unmodifiableSet(tipos);
    }

    /**
     * Determina se um tipo de dados pode ser representado em uma requisio
     * HTTP.
     * <br />
     * Os tipos de dados possveis de serem representados em uma requisio HTTP
     * so {@linkplain String}s, nmeros e as respectivas colees e arrays.
     * @param tipo O tipo de dados a ser determinado se pode ser representado em
     * uma requisio HTTP.
     * @return Se  possvel representar.
     */
    public static boolean tipoRequest(Type tipo) {
        // Se for uma classe apenas, basta verificar se  um tipo fcil de
        // resolver ou um array deste tipo.
        if (tipo instanceof Class) {
            Class<?> c = (Class<?>) tipo;
            return (tipoURL(c) || (c.isArray() && tipoURL(c.getComponentType())));
        }

        // No  uma classe. Se no for um tipo parametrizvel tambm, ento
        // no  um tipo que d para ser resolvido ( um wildcard ou uma
        // varivel de tipo).
        if (!(tipo instanceof ParameterizedType)) return false;

        // S conseguimos reolver List<Foo>, Set<Foo> e similares. Se no tiver
        // um nico parmetro de tio, ento no  nada disso.
        ParameterizedType pt = (ParameterizedType) tipo;
        Type[] params = pt.getActualTypeArguments();
        if (params.length != 1) return false;

        // O parmetro tem que ser um tipo fcil.
        Type param = params[0];
        if (!tipoURL(param)) return false;

        // O tipo tem que ser Iterable, Collection, List, Set ou SortedSet.
        return (COLECOES_POSSIVEIS.contains(pt.getRawType()));
    }

    public static final class StringParseException extends Exception {

        private static final long serialVersionUID = 7598456287025369274L;

        public StringParseException(String mensagem) {
            super(mensagem);
        }
    }

    public static Object representarComoURL(Type tipo, String valor) throws StringParseException {
        if (tipo == null) throw new IllegalArgumentException();
        if (!tipoURL(tipo)) throw new IllegalArgumentException();
        Class<?> classe = (Class<?>) tipo;
        try {
            if (valor == null) return null;
            if (classe == String.class) return valor;
            if (classe.isPrimitive()) classe = EMPACOTADORAS.get(classe);
            if (classe == Character.class) {
                if (valor.length() != 1) {
                    throw new StringParseException("A String \"" + valor + "\" tem que ter 1 caractere.");
                }
                return classe.cast(valor.charAt(0));
            }

            if (!MathUtils.numeroValido(valor)) {
                throw new StringParseException("\"" + valor + "\" no  um nmero vlido.");
            }
            if (classe == Integer.class) return Integer.parseInt(valor);
            if (classe == Long.class) return Long.parseLong(valor);
            if (classe == Short.class) return Short.parseShort(valor);
            if (classe == Byte.class) return Byte.parseByte(valor);
            if (classe == BigInteger.class) return classe.cast(new BigInteger(valor));
            if (classe == BigDecimal.class) return classe.cast(new BigDecimal(valor));
            if (classe == Float.class) return Float.parseFloat(valor);
            if (classe == Double.class) return Double.parseDouble(valor);
        } catch (NumberFormatException e) {
            throw new StringParseException("Xiii");
        }
        throw new AssertionError();
    }

    public static Object representarComoRequest(Type tipo, String[] valor) throws StringParseException {
        if (tipoURL(tipo)) {
            if (valor == null || valor.length == 0) return null;
            if (valor.length > 1) throw new IllegalArgumentException("O tamanho o array deveria ser 1.");
            return representarComoURL((Class<?>) tipo, valor[0]);
        }
        if (!tipoRequest(tipo)) throw new IllegalArgumentException();
        if (valor == null || valor.length == 0) return null;
        if (tipo instanceof Class) {
            Class<?> ctipo = (Class<?>) tipo;
            Class<?> componente = ctipo.getComponentType();
            Object[] resposta = (Object[]) java.lang.reflect.Array.newInstance(componente, valor.length);
            for (int i = 0; i < valor.length; i++) {
                resposta[i] = representarComoURL(componente, valor[i]);
            }
            return resposta;
        }
        ParameterizedType t = (ParameterizedType) tipo;
        Class<?> tipoLista = (Class<?>) t.getRawType();
        Class<?> tipoDado = (Class<?>) t.getActualTypeArguments()[0];
        @SuppressWarnings("unchecked") Collection<Object> lista =
                (tipoLista == Iterable.class || tipoLista == Collection.class || tipoLista == List.class)
                ? new ArrayList<Object>(valor.length)
                : (tipoLista == Set.class)
                ? new HashSet<Object>(valor.length)
                : new TreeSet<Object>(new HashSet<Object>(valor.length));
        for (String v : valor) {
            lista.add(representarComoURL(tipoDado, v));
        }
        return checked(lista, tipoDado, tipoLista);
    }

    @SuppressWarnings("unchecked")
    private static Collection<?> checked(Collection<?> lista, Class<?> classe, Class<?> tipo) {
        if (tipo == Iterable.class || tipo == Collection.class || tipo == List.class) {
            return Collections.checkedList((List) lista, classe);
        }
        if (tipo == Set.class) {
            return Collections.checkedSet((Set) lista, classe);
        }
        return Collections.checkedSortedSet((SortedSet) lista, classe);
    }
}
