/*
 * 16/03/2009, 11:25.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.util;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Classe que serializa objetos em formato JSON.
 * @author Victor Williams Stafusa da Silva
 */
public class JsonObject {

    /**
     * Erro resultante do uso do serializador em classes que no usam a anotao
     * {@linkplain JsonMarshal} corretamente.
     */
    public static class JsonSerializerError extends Error {

        private static final long serialVersionUID = -217508291560979309L;

        /**
         * Cria o erro indicando qual mtodo no pode ser usado na serializao
         * por ter uma assinatura inadequada para o uso da anotao
         * {@linkplain JsonMarshal}.
         * @param metodo O mtodo com a assinatura inadequada.
         */
        public JsonSerializerError(Method metodo) {
            super("O mtodo " + metodo.getDeclaringClass() + "." + metodo.getName() + " no tem a assinatura apropriada.");
        }

        /**
         * Cria o erro indicando qual mtodo que no pde ser usado na
         * serializao por ter lanado uma exceo ao ser invocado
         * {@linkplain JsonMarshal}.
         * @param metodo O mtodo que lanou a exceo.
         * @param causa A exceo lanada.
         */
        public JsonSerializerError(Method metodo, Throwable causa) {
            super("O mtodo " + metodo.getDeclaringClass() + "." + metodo.getName() + " lanou uma exceo: " + causa.toString(), causa);
        }
    }

    private final Map<String, Object> filhos;

    private final Map<Class<?>, List<String>> naoVisitar;

    public static JsonObject serializar(Object obj) {
        return serializar(obj, null);
    }

    public static JsonObject serializar(Object obj, Map<Class<?>, List<String>> naoVisitar) {
        if (obj == null) {
            throw new IllegalArgumentException("O objeto no pode ser nulo.");
        }
        if (obj instanceof JsonObject) {
            return (JsonObject) obj;
        }

        JsonObject json = new JsonObject(naoVisitar);
        if (obj instanceof Map) {
            json.construir((Map<?, ?>) obj);
        } else if (classeSerializavelJson(obj.getClass())) {
            json.construir(obj);
        } else {
            throw new IllegalArgumentException("O objeto da classe " + obj.getClass().getName() + " no pode ser serializado.");
        }
        return json;
    }

    public JsonObject() {
        this(null);
    }

    public JsonObject(Map<Class<?>, List<String>> naoVisitar) {
        this.filhos = new HashMap<String, Object>();
        this.naoVisitar = naoVisitar != null ? naoVisitar : new HashMap<Class<?>, List<String>>();
    }

    public void naoVisitar(Class<?> c, List<String> campos) {
        this.naoVisitar.put(c, campos);
    }

    public void naoVisitar(Class<?> c, String... campos) {
        naoVisitar(c, Arrays.asList(campos));
    }

    public Object get(String chave) {
        return filhos.get(chave);
    }

    public void remove(String chave) {
        filhos.remove(chave);
    }

    public void put(String chave, Object filho) {
        filhos.put(chave, filho);
    }

    /**
     * Descobre se uma classe est anotada com a anotao
     * {@linkplain JsonMarshal}. Uma classe  considerada anotada com
     * {@linkplain JsonMarshal} por este mtodo, mesmo se ela herdar de alguma
     * classe que esteja anotada com {@linkplain JsonMarshal} ou se ela
     * implementar alguma interface anotada com {@linkplain JsonMarshal}, e no
     * apenas se ela prpria estiver anotada com {@linkplain JsonMarshal}.
     * @param classe A classe a ser verificada.
     * @return Se a classe est anotada com {@linkplain JsonMarshal}.
     */
    private static boolean classeSerializavelJson(Class<?> classe) {
        // Verifica se a anotao est na prpria classe ou em alguma
        // superclasse.
        for (Class<?> c = classe; c != null; c = c.getSuperclass()) {
            if (c.isAnnotationPresent(JsonMarshal.class)) return true;
        }

        // Verifica se a anotao est em alguma interface implementada.
        for (Class<?> c : classe.getInterfaces()) {
            if (c.isAnnotationPresent(JsonMarshal.class)) return true;
        }

        // No achou a anotao. Desiste.
        return false;
    }

    /**
     * Descobre se um mtodo est anotado com a anotao
     * {@linkplain JsonMarshal}. Um dado mtodo  considerado anotado com
     * {@linkplain JsonMarshal} por este mtodo, mesmo se ele estiver
     * sobrescrever um mtodo anotado com {@linkplain JsonMarshal}, mesmo se
     * estiver em uma interface implementada.
     * @param metodo O mtodo a ser verificado.
     * @return Se o mtodo est anotado com {@linkplain JsonMarshal}.
     */
    private static JsonMarshal metodoSerializavelJson(Method metodo) {

        // Verifica se o mtodo est anotado, ou se algum dos mtodos
        // sobrescritos na superclasse est.
        for (Method m = metodo; m != null; ) {

            // Verifica a anotao no mtodo.
            JsonMarshal js = m.getAnnotation(JsonMarshal.class);
            if (js != null) return js;

            // Move para a superclasse, caso exista.
            Class<?> c = m.getDeclaringClass().getSuperclass();
            if (c == null) break;

            try {
                // Obtm o mtodo sobrescrito.
                m = c.getMethod(metodo.getName(), metodo.getParameterTypes());
            } catch (NoSuchMethodException e) {
                break; // No h mtodo sobrescrito com a anotao.
            }
        }

        Class<?> classe = metodo.getDeclaringClass();
        for (Class<?> c : classe.getInterfaces()) {
            try {
                // Obtm o mtodo da interface.
                Method m = c.getMethod(metodo.getName(), metodo.getParameterTypes());

                // Verifica a anotao no mtodo.
                JsonMarshal js = m.getAnnotation(JsonMarshal.class);
                if (js != null) return js;
            } catch (NoSuchMethodException e) {
                // O mtodo no est na interface. Ignora a exceo e continua.
            }
        }

        // No achou. Desiste.
        return null;
    }

    /**
     * Serializa este objeto em uma {@linkplain String} em formato JSON.
     * @return A {@linkplain String} em formato JSON equivalente ao objeto
     * serializado.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        boolean primeiro = true;

        for (Map.Entry<String, Object> filho : filhos.entrySet()) {
            if (primeiro) {
                primeiro = false;
            } else {
                sb.append(", ");
            }
            sb.append('\"').append(filho.getKey()).append("\": ");
            concatenar(sb, filho.getValue());
        }
        return sb.append("}").toString();
    }

    /**
     * Mantm uma lista global (de cache) listando quais mtodos de cada classe
     * esto anotados com {@linkplain JsonMarshal}.
     */
    private static final Map<Class<?>, List<Method>> METODOS_JSON =
            new ConcurrentHashMap<Class<?>, List<Method>>();

    /**
     * Obtm os mtodos que contm a anotao {@linkplain JsonMarshal} para a
     * classe dada.
     * @param classe A classe da qual deseja-se saber quais so os mtodos que
     * esto anotados com {@linkplain JsonMarshal}.
     * @return Uma lista com mtodos teis na serializao.
     */
    private static List<Method> metodosJson(Class<?> classe) {
        // Primeiro procura na cache.
        List<Method> metodos = METODOS_JSON.get(classe);
        if (metodos != null) return metodos;

        metodos = new ArrayList<Method>(20);

        // Itera pelos mtodos PBLICOS da classe dada, procurando os que tem
        // a anotao JsonMarshal.
        Method[] metodosClasse = classe.getMethods();
        for (Method m : metodosClasse) {
            // Se a anotao JsonMarshal no est presente, pula este mtodo.
            if (metodoSerializavelJson(m) == null) continue;

            // Achou um mtodo pertinente. O adiciona.
            metodos.add(m);
        }

        // A lista est pronta, ento de agora em diante, ela no pode ser
        // alterada.
        metodos = Collections.unmodifiableList(metodos);

        // Coloca os mtodos encontrados na cache e os retorna.
        METODOS_JSON.put(classe, metodos);
        return metodos;
    }

    private void construir(Map<?, ?> objeto) {
        for (Map.Entry<?, ?> e : objeto.entrySet()) {
            put(e.getKey().toString(), e.getValue());
        }
    }

    private void construir(Object objeto) {
        // Obtm os mtodos que tm a anotao JsonMarshal.
        List<Method> metodos = metodosJson(objeto.getClass());

        for (Method m : metodos) {

            // Descobre o nome do campo.
            String nomeCampo = nomePropriedade(m);

            // Se o mtodo no for algum mtodo que devamos visitar, pule!
            List<String> lista = naoVisitar.get(m.getDeclaringClass());
            if (lista != null && lista.contains(nomeCampo)) continue;

            // O mtodo  pblico, mas talvez a classe onde ele  declarado no
            // seja.
            m.setAccessible(true);

            // Invoca o mtodo para obter o valor a ser serializado.
            Object resultado = null;
            try {
                resultado = m.invoke(objeto);
            } catch (IllegalAccessException e) {
                // No deveria ocorrer nunca, pois o mtodo  pblico.
                throw new AssertionError(e);
            } catch (InvocationTargetException e) {
                // O mtodo lanou uma exceo. Abortar a serializao.
                throw new JsonSerializerError(m, e.getCause());
            }

            // Coloca o resultado do mtodo no objeto JSON em construo.
            put(nomeCampo, resultado);
        }
    }

    /** Usado para formatar datas em JSON. */
    private static final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss-SSSS");

    /**
     * Formata uma data para ser colocada posteriormente em formato JSON.
     * @param data A data a ser formatada.
     * @return A data serializada.
     */
    private static synchronized String formatar(Date data) {
        return sdf.format(data);
    }

    /**
     * Serializa um objeto em formato JSON e concatena o resultado em um
     * {@linkplain StringBuilder}, que assume-se j conter previamente o nome do
     * campo aonde o objeto ser colocado.
     * @param sb O {@linkplain StringBuilder} aonde o objeto ser concatenado.
     * @param resultado O objeto a ser concatenado.
     */
    private void concatenar(StringBuilder sb, Object resultado) {
        // Caso 1: O objeto  null.
        if (resultado == null) {
            sb.append("null");

        // Caso 2: O objeto  uma String. Faz o escape, formata e o concatena.
        } else if (resultado instanceof String) {
            sb.append('\"').append(StringUtils.jsonFormat((String) resultado)).append('\"');

        // Caso 3: O objeto  um Date. Faz a formatao e o concatena.
        } else if (resultado instanceof Date) {
            sb.append('\"').append(formatar((Date) resultado)).append('\"');

        // Caso 4: O objeto  uma coleo, concatena um array JSON contendo cada
        // um dos elementos.
        } else if (resultado instanceof Iterable) {
            concatenarCollection(sb, (Iterable<?>) resultado);

        // Caso 5: O objeto  um array, faz o mesmo tratamento que foi feito no
        // caso nmero 4.
        } else if (resultado.getClass().isArray()) {
            concatenarCollection(sb, Arrays.asList((Object[]) resultado));

        // Caso 6: O objeto  alguma coisa que contm a anotao @JsonMarshal,
        // ento serializa esse objeto tambm.
        } else if (classeSerializavelJson(resultado.getClass())) {
            sb.append(JsonObject.serializar(resultado, naoVisitar).toString());

        // Caso 7: O objeto  um Map, considera-o como se fosse um objeto e
        // concatena-o.
        } else if (resultado instanceof Map) {
            concatenarMap(sb, (Map<?, ?>) resultado);

        // Caso 8: O objeto  alguma outra coisa qualquer. Chama o toString() e
        // confia cegamente que vai dar certo, sem fazer nenhum tipo de escape
        // ou tratamento.
        } else {
            sb.append(resultado.toString());
        }
    }

    /**
     * Serializa uma coleo em formato JSON e concatena o resultado, juntamente
     * com cada elemento em um {@linkplain StringBuilder}, que assume-se j
     * conter previamente o nome do campo aonde o objeto ser colocado.
     * @param sb O {@linkplain StringBuilder} aonde o objeto ser concatenado.
     * @param resultado A coleo a ser concatenada.
     */
    private void concatenarCollection(StringBuilder sb, Iterable<?> resultado) {
        sb.append('[');
        boolean primeiro = true;
        for (Object o : resultado) {
            if (primeiro) {
                primeiro = false;
            } else {
                sb.append(", ");
            }
            concatenar(sb, o);
        }
        sb.append(']');
    }

    /**
     * Serializa um {@linkplain Map} em formato JSON e concatena o resultado,
     * juntamente com cada elemento em um {@linkplain StringBuilder}, que
     * assume-se j conter previamente o nome do campo aonde o objeto ser
     * colocado.
     * @param sb O {@linkplain StringBuilder} aonde o objeto ser concatenado.
     * @param resultado O Map a ser concatenada.
     */
    private void concatenarMap(StringBuilder sb, Map<?, ?> resultado) {
        sb.append('{');
        boolean primeiro = true;
        for (Map.Entry<?, ?> e : resultado.entrySet()) {
            if (primeiro) {
                primeiro = false;
            } else {
                sb.append(", ");
            }
            sb.append("\"")
                    .append(StringUtils.jsonFormat(e.getKey().toString()))
                    .append("\": ");
            concatenar(sb, e.getValue());
        }
        sb.append('}');
    }

    /**
     * Descobre o nome da propriedade de um determinado getter.
     * @param metodo O mtodo cujo nome da propriedade ser determinado
     * @return O nome da propriedade do mtodo.
     * @throws JsonSerializerError Caso o mtodo dado no seja um getter ou no
     * or possvel descobrir o nome da propriedade dele.
     */
    private String nomePropriedade(Method metodo) throws JsonSerializerError {

        // Valida se o mtodo anotado  de instncia, no-void e sem parmetros.
        // Caso no seja, no se trata de um getter, ento lana um erro.
        if (metodo.getParameterTypes().length != 0) throw new JsonSerializerError(metodo);
        if (metodo.getReturnType() == void.class || metodo.getReturnType() == Void.class) throw new JsonSerializerError(metodo);
        if (Modifier.isStatic(metodo.getModifiers())) throw new JsonSerializerError(metodo);

        // Caso o nome da propriedade esteja na anotao, terminamos facilmente.
        JsonMarshal js = metodoSerializavelJson(metodo);
        String nome = js.value();
        if (!"".equals(nome)) return nome;

        // O nome no est na anotao, ento comeamos com o nome do mtodo
        // para deduzir o nome da propriedade.
        nome = metodo.getName();

        // Se o nome do mtodo segue o modelo getXxx, ento descobrimos o nome
        // da propriedade olhando para xxx.
        if (nome.startsWith("get") && Character.isUpperCase(nome.charAt(3))) {
            return new StringBuilder()
                    .append(Character.toLowerCase(nome.charAt(3)))
                    .append(nome.substring(4))
                    .toString();
        }

        // Se o nome do mtodo segue o modelo isXxx e o mtodo retorna boolean,
        // ento descobrimos o nome da propriedade olhando para xxx.
        if ((metodo.getReturnType() == boolean.class || metodo.getReturnType() == Boolean.class)
                && nome.startsWith("is") && Character.isUpperCase(nome.charAt(2))) {
            return new StringBuilder()
                    .append(Character.toLowerCase(nome.charAt(2)))
                    .append(nome.substring(3))
                    .toString();
        }

        // No foi possvel descobrir o nome da propriedade. Desiste.
        throw new JsonSerializerError(metodo);
    }
}