/*
 * 08/03/2009, 15:05.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.usuario.cadastro;

import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.util.JsonMarshal;
import br.com.simuquiz.util.StringUtils;
import java.io.Serializable;
import static br.com.simuquiz.util.StringUtils.*;

/**
 * Classe responsvel por representar a funcionalidade de cadastrar usurio
 * contendo os dados da tela informados pelo usurio, a resposta a ser enviada
 * e os erros que tiverem ocorrido no processo.
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeCadastrarUsuario implements Serializable {

    private static final long serialVersionUID = 3080944437689040882L;

    /** Os dados de entrada digitados/escolhidos pelo usurio. */
    private final Entrada entrada;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** O bean de usurio resultante do processo de cadastro. */
    private final Usuario usuario;

    public enum TiposErros {
        ERRO, NOME, LOGIN, SENHA, CONFIRMACAO_SENHA, EMAIL, CONFIRMACAO_EMAIL
    }

    /**
     * Realiza o processo de cadastro de usurio, que resulta em uma instncia
     * desta classe contendo os dados informados para a realizao do cadastro,
     * os dados referentes ao resultado do cadastro e tambm os problemas
     * ocorridos.
     * @param login O contedo do campo "login" informado pelo usurio, ou seja,
     * o login a ser cadastrado.
     * @param senha O contedo do campo "senha" informada pelo usurio, ou seja,
     * a senha a ser cadastrada.
     * @param confirmacaoSenha O contedo do campo de "confirmao de senha"
     * informado pelo usurio.
     * @param email O contedo do campo "e-mail" informado pelo usurio, ou
     * seja, o endereo de e-mail do usurio.
     * @param confirmacaoEmail O contedo do campo "confirmao do e-mail"
     * informado pelo usurio.
     * @param nome O contedo do campo "nome" informado pelo usurio, ou seja, o
     * nome do usurio.
     * @return O bean da funcionalidade contendo os parmetros informados, o
     * resultado obtido e os problemas ocorridos.
     */
    public static FuncionalidadeCadastrarUsuario incluir(String login, String senha, String confirmacaoSenha, String email, String confirmacaoEmail, String nome) {
    	return new FuncionalidadeCadastrarUsuario(new IncluirUsuarioStrategy(new Entrada(nome, login, senha, confirmacaoSenha, email, confirmacaoEmail)));
    }

    /**
     * Realiza o processo de atualizao de usurio. O resultado  uma instncia
     * desta classe contendo os dados informados para a realizao do cadastro,
     * os dados referentes ao resultado do cadastro e tambm os problemas
     * ocorridos.
     * @param usuario O bean que representa o usurio que ser atualizado no
     * banco de dados.
     * @param login O contedo do campo "login" informado pelo usurio, ou seja,
     * o login a ser cadastrado.
     * @param senha O contedo do campo "senha" informada pelo usurio, ou seja,
     * a senha a ser cadastrada.
     * @param confirmacaoSenha O contedo do campo de "confirmao de senha"
     * informado pelo usurio.
     * @param email O contedo do campo "e-mail" informado pelo usurio, ou
     * seja, o endereo de e-mail do usurio.
     * @param confirmacaoEmail O contedo do campo "confirmao do e-mail"
     * informado pelo usurio.
     * @param nome O contedo do campo "nome" informado pelo usurio, ou seja, o
     * nome do usurio.
     * @return O bean da funcionalidade contendo os parmetros informados, o
     * resultado obtido e os problemas ocorridos.
     */
    public static FuncionalidadeCadastrarUsuario alterar(Usuario usuario, String login, String senha, String confirmacaoSenha, String email, String confirmacaoEmail, String nome) {
        return new FuncionalidadeCadastrarUsuario(new AlterarUsuarioStrategy(usuario, new Entrada(nome, login, senha, confirmacaoSenha, email, confirmacaoEmail)));
    }

    /**
     * Realiza o processo de cadastro ou atualizao de usurio, de acordo com
     * a estrategia especificada.
     * O resultado  uma instncia desta classe contendo os dados informados
     * para a realizao do cadastro, os dados referentes ao resultado do
     * cadastro e tambm os problemas ocorridos.
     * @param strategy Estratgia para o cadastro, pode ser e incluso, ou de
     * alterao.
     */
    private FuncionalidadeCadastrarUsuario(Strategy strategy) {

        // Grava os parmetros informados.
    	this.entrada = strategy.getEntrada();

        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);

        // Faz todas as validaes e lista os problemas encontrados.
        if (vazio(entrada.getLogin())) {
            problemas.adicionarProblema(TiposErros.LOGIN, "O login deve ser informado.");
        }
        if (vazio(entrada.getNome())) {
            problemas.adicionarProblema(TiposErros.NOME, "O nome do usurio deve ser informado.");
        }

        if (strategy.exigirVerificarSenha()) {
            if (vazio(entrada.getSenha())) {
                problemas.adicionarProblema(TiposErros.SENHA, "A senha deve ser informada.");
            }
            if (vazio(entrada.getConfirmacaoSenha())) {
                problemas.adicionarProblema(TiposErros.CONFIRMACAO_SENHA, "O campo \"confirmao da senha\" deve ser informado.");
            }
            if (!StringUtils.equals(entrada.getSenha(), entrada.getConfirmacaoSenha())) {
                problemas.adicionarProblema(TiposErros.CONFIRMACAO_SENHA, "As senhas informadas no coincidem.");
            }
        }

        if (vazio(entrada.getEmail())) {
            problemas.adicionarProblema(TiposErros.EMAIL, "O endereo de e-mail deve ser informado.");
        }
        if (vazio(entrada.getConfirmacaoEmail())) {
            problemas.adicionarProblema(TiposErros.CONFIRMACAO_EMAIL, "O campo \"confirmao do e-mail\" deve ser informado.");
        }
        if (!StringUtils.equals(entrada.getEmail(), entrada.getConfirmacaoEmail())) {
            problemas.adicionarProblema(TiposErros.CONFIRMACAO_EMAIL, "Os endereos de e-mail informados no coincidem.");
        }
        if (!emailValido(entrada.getEmail())) {
            problemas.adicionarProblema(TiposErros.CONFIRMACAO_EMAIL, "O endereo de e-mail no  vlido.");
        }

        // Em caso de no haver problemas, verifica se o login j existe.
        if (problemas.isVazio() && strategy.exigirVerificarLoginExistente()) {
            try {
                Usuario x = Usuario.pesquisarPorLogin(entrada.getLogin());
                if (x != null) problemas.adicionarProblema(TiposErros.LOGIN, "Este login j existe.");
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }

        // Em caso de no haver problemas, cadastra o usurio.
        Usuario u = null;
        if (problemas.isVazio()) {
            try {
            	// Faz o cadastro do usurio (seja isso um INCLUDE ou um
                // UPDATE).
            	u = strategy.cadastrar();
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
            // TODO: Acrescentar envio de e-mail de confirmao.
        }
        this.usuario = u;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    /**
     * Retorna as informaes de entrada digitadas/escolhidas pelo usurio.
     * @return As informaes dadas pelo usurio.
     */
    @JsonMarshal
    public Entrada getEntrada() {
        return entrada;
    }

    /**
     * Indica se o processo de cadastro de usurio foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    /**
     * Retorna o usurio resultante do processo de cadastro.
     * @return O bean do usurio resultante do processo de cadastro ou
     * {@code null} em caso de o cadastro no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de cadastro de
     * usurio.
     * @return Os problemas resultantes do processo de cadastro.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }
}
