/*
 * 10/03/2009, 05:35.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.usuario;

import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import static br.com.simuquiz.util.StringUtils.*;

/**
 * Classe responsvel por representar a funcionalidade de login, contendo os
 * dados da tela informados pelo usurio, a resposta a ser enviada e os erros
 * que tiverem ocorrido no processo.
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeLogin implements Serializable {

    private static final long serialVersionUID = -3371738396663637043L;

    /** O login escolhido pelo usurio. */
    private final String login;

    /** A senha escolhida pelo usurio. */
    private final String senha;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** O bean de usurio resultante de um login bem sucedido. */
    private final Usuario usuario;

    public enum TiposErros { ERRO, LOGIN, SENHA }

    /**
     * Realiza o login do usurio, que resulta em uma instncia desta classe
     * contendo o login e a senha informada, o usurio resultante do processo
     * de login e tambm os problemas ocorridos.
     * @param login O contedo do campo "login" informado pelo usurio.
     * @param senha O contedo do campo "senha" informado pelo usurio.
     */
    public FuncionalidadeLogin(String login, String senha) {
        // Grava os parmetros informados.
        this.login = login;
        this.senha = senha;

        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);

        // Faz todas as validaes e lista os problemas encontrados.
        if (vazio(login)) {
            problemas.adicionarProblema(TiposErros.LOGIN, "O login deve ser informado.");
        }
        if (vazio(senha)) {
            problemas.adicionarProblema(TiposErros.SENHA, "A senha deve ser informada.");
        }

        // Em caso de no haver problemas, tenta fazer o login.
        Usuario u = null;
        if (problemas.isVazio()) {
            try {
                u = Usuario.login(login, senha);
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }

            // Verifica se o login foi bem sucedido.
            if (u == null) {
                problemas.adicionarProblema(TiposErros.LOGIN, "Login e/ou senha incorretos.");
            }
        }
        usuario = u;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    /**
     * Retorna o login digitado pelo usurio.
     * @return O login digitado pelo usurio.
     */
    @JsonMarshal
    public String getLogin() {
        return login;
    }

    /**
     * Retorna a senha digitada pelo usurio.
     * @return A senha digitada pelo usurio.
     */
    public String getSenha() {
        return senha;
    }

    /**
     * Indica se o processo de login foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    /**
     * Retorna o usurio encontrado como resultado do processo de login.
     * @return O bean do usurio resultante do processo de login ou {@code null}
     * em caso de o login no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de login.
     * @return Os problemas resultantes do processo de login.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }
}
