/*
 * 12/03/2009, 07:16.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.exame;

import br.com.simuquiz.entidades.Exame;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import static br.com.simuquiz.util.StringUtils.*;

/**
 * Classe responsvel por representar a funcionalidade de cadastrar exame
 * contendo os dados da tela informados pelo usurio, a resposta a ser enviada
 * e os erros que tiverem ocorrido no processo.
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeCadastrarExame implements Serializable {

    private static final long serialVersionUID = -6125250553936505404L;

    /** Usurio que est logado. */
    private final Usuario usuario;

    /** Nome do exame informado pelo usurio. */
    private final String nome;

    /** Descrio curta informada pelo usurio. */
    private final String descricaoCurta;

    /** Descrio completa informada pelo usurio. */
    private final String descricaoCompleta;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** O bean de exame resultante do processo de cadastro. */
    private final Exame exame;

    public enum TiposErros { ERRO, NOME, DESCRICAO_CURTA, DESCRICAO_COMPLETA }

    public FuncionalidadeCadastrarExame(Usuario usuario, String nome, String descricaoCurta, String descricaoCompleta) {
        if (usuario == null) {
            throw new IllegalArgumentException();
        }

        this.usuario = usuario;
        this.nome = nome;
        this.descricaoCurta = descricaoCurta;
        this.descricaoCompleta = descricaoCompleta;

        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);

        // Faz todas as validaes e lista os problemas encontrados.
        if (vazio(nome)) {
            problemas.adicionarProblema(TiposErros.NOME, "O nome do exame deve ser informado.");
        }

        // Em caso de no haver problemas, cadastra o exame.
        Exame e = null;
        if (problemas.isVazio()) {
            try {
                e = usuario.novoExame(nome, descricaoCurta, descricaoCompleta);
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }
        exame = e;

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    /**
     * Retorna o bean do usurio logado no sistema, que ser o proprietrio do
     * exame criado por esta funcionalidade.
     * @return O bean do usurio logado no sistema.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    /**
     * Retorna o nome do exame informado pelo usurio.
     * @return O nome do exame informado pelo usurio.
     */
    @JsonMarshal
    public String getNome() {
        return nome;
    }

    /**
     * Retorna a descrio curta do exame informada pelo usurio.
     * @return A descrio curta do exame informada pelo usurio.
     */
    @JsonMarshal
    public String getDescricaoCurta() {
        return descricaoCurta;
    }

    /**
     * Retorna a descrio completa do exame informada pelo usurio.
     * @return A descrio completa do exame informada pelo usurio.
     */
    @JsonMarshal
    public String getDescricaoCompleta() {
        return descricaoCompleta;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de cadastro de exame.
     * @return Os problemas resultantes do processo de cadastro.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }

    /**
     * Indica se o processo de cadastro de exame foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    /**
     * Retorna o exame resultante do processo de cadastro.
     * @return O bean do usurio resultante do processo de cadastro ou
     * {@code null} em caso de o cadastro no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Exame getExame() {
        return exame;
    }

    @JsonMarshal
    public double getRanking() {
        return -1;
    }
}
