/*
 * 13/03/2009, 06:45.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades.exame;

import br.com.simuquiz.entidades.Exame;
import br.com.simuquiz.entidades.Usuario;
import br.com.simuquiz.funcionalidades.RelatorioProblemas;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import static br.com.simuquiz.util.StringUtils.*;

/**
 * Classe responsvel por representar a funcionalidade de alterar um exame
 * existente, contendo os dados da tela informados pelo usurio, a resposta a
 * ser enviada e os erros que tiverem ocorrido no processo.
 *
 * OBSERVAO: FUNCIONALIDADE INCOMPLETA!
 *
 * @author Victor Williams Stafusa da Silva
 */
@JsonMarshal
public final class FuncionalidadeAlterarExame implements Serializable {

    private static final long serialVersionUID = 1681800822028671381L;

    /** Usurio que est logado. */
    private final Usuario usuario;

    /** Id do exame informado pelo usurio. */
    private final Long idExame;

    /** Novo nome do exame informado pelo usurio. */
    private final String nome;

    /** Nova descrio curta informada pelo usurio. */
    private final String descricaoCurta;

    /** Nova descrio completa informada pelo usurio. */
    private final String descricaoCompleta;

    /** Relao de problemas ocorridos. */
    private final RelatorioProblemas<TiposErros> problemas;

    /** O bean de exame resultante do processo de alterao. */
    private final Exame exame;

    public enum TiposErros { ERRO, EXAME, NOME, DESCRICAO_CURTA, DESCRICAO_COMPLETA }

    public FuncionalidadeAlterarExame(Usuario usuario, Long idExame, String nome, String descricaoCurta, String descricaoCompleta) {
        if (usuario == null) {
            throw new IllegalArgumentException();
        }

        this.idExame = idExame;
        this.usuario = usuario;
        this.nome = nome;
        this.descricaoCurta = descricaoCurta;
        this.descricaoCompleta = descricaoCompleta;

        this.problemas = new RelatorioProblemas<TiposErros>(TiposErros.class);

        // Faz todas as validaes e lista os problemas encontrados.
        if (idExame == null) {
            problemas.adicionarProblema(TiposErros.EXAME, "O exame deve ser informado.");
        }
        if (vazio(nome)) {
            problemas.adicionarProblema(TiposErros.NOME, "O nome do exame deve ser informado.");
        }

        // Em caso de no haver problemas, obtm o exame.
        Exame e = null;
        if (problemas.isVazio()) {
            try {
                e = Exame.pesquisarPorId(idExame);
                if (e == null) problemas.adicionarProblema(TiposErros.EXAME, "O exame no foi encontrado.");
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }
        exame = e;

        // Em caso de no haver problemas, verifica se o usurio  proprietrio do exame.
        if (problemas.isVazio() && exame.getProprietario() != usuario) {
            problemas.adicionarProblema(TiposErros.EXAME, "Apenas o proprietrio do exame pode alter-lo.");
        }

        // Finalmente, se nada deu errado, faz as alteraes.
        if (problemas.isVazio()) {
            try {
                exame.setNome(nome);
                exame.setDescricaoCurta(descricaoCurta);
                exame.setDescricaoCompleta(descricaoCompleta);
                exame.salvar();
            } catch (Throwable t) {
                problemas.adicionarProblema(TiposErros.ERRO, t);
                t.printStackTrace();
                // TODO: Acrescentar log.
            }
        }

        // Finaliza o processo e se torna apenas um bean.
        problemas.fechar();
    }

    /**
     * Retorna o bean do usurio logado no sistema, que ser o proprietrio do
     * exame criado por esta funcionalidade.
     * @return O bean do usurio logado no sistema.
     */
    @JsonMarshal
    public Usuario getUsuario() {
        return usuario;
    }

    @JsonMarshal
    public Long getIdExame() {
        return idExame;
    }

    /**
     * Retorna o nome do exame informado pelo usurio.
     * @return O nome do exame informado pelo usurio.
     */
    @JsonMarshal
    public String getNome() {
        return nome;
    }

    /**
     * Retorna a descrio curta do exame informada pelo usurio.
     * @return A descrio curta do exame informada pelo usurio.
     */
    @JsonMarshal
    public String getDescricaoCurta() {
        return descricaoCurta;
    }

    /**
     * Retorna a descrio completa do exame informada pelo usurio.
     * @return A descrio completa do exame informada pelo usurio.
     */
    @JsonMarshal
    public String getDescricaoCompleta() {
        return descricaoCompleta;
    }

    /**
     * Retorna a lista de problemas ocorridos no processo de cadastro de exame.
     * @return Os problemas resultantes do processo de cadastro.
     */
    @JsonMarshal
    public RelatorioProblemas getProblemas() {
        return problemas;
    }

    /**
     * Indica se o processo de cadastro de exame foi bem sucedido.
     * @return {@code true} se o processo foi bem sucedido, {@code false} em
     * caso contrrio.
     */
    @JsonMarshal
    public boolean isSucesso() {
        return problemas.isVazio();
    }

    /**
     * Retorna o exame resultante do processo de cadastro.
     * @return O bean do usurio resultante do processo de cadastro ou
     * {@code null} em caso de o cadastro no tiver sido bem sucedido.
     */
    @JsonMarshal
    public Exame getExame() {
        return exame;
    }

    @JsonMarshal
    public double getRanking() {
        return exame == null ? -1 : exame.ranking();
    }
}
