/*
 * 08/03/2009, 15:55.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.funcionalidades;

import java.io.Serializable;
import static br.com.simuquiz.util.StringUtils.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

/**
 * Classe responsvel por representar uma coleo de problemas ocorridos na
 * execuo de algum processo.
 * @param <E> Classe {@linkplain Enum} cujo elementos representam os tipos de
 * problemas a serem aceitos.
 * @author Victor Williams Stafusa da Silva
 */
public final class RelatorioProblemas<E extends Enum<E>> implements Serializable {

    private static final long serialVersionUID = -2376574288311674549L;

    /** Relao dos problemas ocorridos. */
    private final Map<E, List<String>> problemas;

    /** Relao dos tipos de problemas possveis. */
    private final Class<E> classe;

    /** Cache do valor retornado pelo mtodo {@linkplain #toString()}. */
    private transient String toStringCache;

    /** Flag que indica se este objeto pode ser alterado. */
    private boolean fechado;

    /** Flag que indica se algum erro j foi inserido ou no. */
    private boolean vazio;

    /**
     * Cria uma lista de problemas vazia.
     * @param classe Classe {@linkplain Enum} cujo elementos representam os
     * tipos de problemas a serem aceitos.
     */
    public RelatorioProblemas(Class<E> classe) {

        // Deve existir pelo menos um tipo de problema.
        if (classe == null) {
            throw new IllegalArgumentException("Deve haver um tipo.");
        }

        //  possvel que o tipo no seja um enum, vez que o compilador gera
        // uma warning unchecked em vez de um erro de compilao quando um tipo
        // bruto possivelmente incompatvel de classe  usado.
        if (!classe.isEnum()) {
            throw new IllegalArgumentException("Tipo invlido. Deve ser um enum.");
        }

        // Inicializa alguns atributos.
        this.classe = classe;
        this.problemas = new EnumMap<E, List<String>>(classe);
        this.fechado = false;
        this.vazio = true;

        // Acrescenta cada um dos tipos de problema, verificado se so vlidos.
        for (E elem : classe.getEnumConstants()) {
            problemas.put(elem, new ArrayList<String>());
        }
    }

    /**
     * Verifica se o tipo informado  compatvel com esta classe.
     * <br />
     *  possvel que o tipo seja de uma classe enum errada, vez que o
     * compilador gera uma warning unchecked em vez de um erro de compilao
     * quando um tipo bruto possivelmente incompatvel  usado.
     * @param tipo O tipo informado.
     * @throws IllegalArgumentException Se o tipo for {@code null}.
     * @throws ClassCastException Se o parmetro {@code tipo} no corresponder
     * ao tipo desta classe.
     */
    private void verificarTipoValido(E tipo) {
        if (tipo == null) {
            throw new IllegalArgumentException("O tipo deve ser especificado.");
        }

        if (!classe.isInstance(tipo)) {
            throw new ClassCastException("Tipo incorreto. Recebeu "
                    + tipo.getClass().getName()
                    + " mas deveria ser "
                    + classe.getClass().getName()
                    + ".");
        }
    }

    /**
     * Adiciona um problema a ser mostrado ao usurio.
     * @param tipo O tipo do problema a ser adicionado.
     * @param descricao A descrio do problema a ser adicionado.
     * @throws IllegalArgumentException Se qualquer um dos parmetro for
     * {@code null} ou vazio.
     * @throws IllegalStateException Se a lista de problemas j tiver sido
     * fechada.
     * @throws ClassCastException Se o parmetro {@code tipo} no corresponder
     * ao tipo desta classe.
     */
    public void adicionarProblema(E tipo, String descricao) {
        // Verificaes dos parmetros.
        verificarTipoValido(tipo);

        if (descricao == null || descricao.equals("")) {
            throw new IllegalArgumentException("A descrio deve ser preenchida.");
        }

        if (fechado) {
            throw new IllegalStateException("A lista de problemas j est fechada.");
        }

        List<String> falhas = problemas.get(tipo);

        // Acrescenta a descrio do problema.
        falhas.add(descricao);
        this.vazio = false;

        // Invalida a cache do toString().
        toStringCache = null;
    }

    /**
     * Adiciona um problema a ser mostrado ao usurio.
     * @param tipo O tipo do problema a ser adicionado.
     * @param erro O erro cuja descrio deve ser adicionada. Caso no tenha
     * descrio, o nome da classe do erro  usada.
     * @throws IllegalArgumentException Se qualquer um dos parmetro for
     * {@code null} ou se o erro no tiver mensagem nele.
     * @throws IllegalStateException Se a lista de problemas j tiver sido
     * fechada.
     * @throws ClassCastException Se o parmetro {@code tipo} no corresponder
     * ao tipo desta classe.
     */
    public void adicionarProblema(E tipo, Throwable erro) {
        if (erro == null) {
            throw new IllegalArgumentException("A descrio deve ser preenchida.");
        }
        String mensagem = erro.getMessage();
        if (mensagem == null || mensagem.equals("")) mensagem = erro.getClass().getName();
        this.adicionarProblema(tipo, mensagem);
    }

    /**
     * Indica se a lista de problemas est vazia ou no.
     * @return {@code true} se a lista de problemas est vazia ou {@code false}
     * em caso contrrio.
     */
    public boolean isVazio() {
        return vazio;
    }

    /**
     * Indica se a lista de problemas est fechada (no pode ser alterada) ou
     * no.
     * @return {@code true} se a lista de problemas est fechada ou
     * {@code false} em caso contrrio.
     */
    public boolean isFechado() {
        return fechado;
    }

    /**
     * Fecha a lista de problemas, no permitindo que ela seja posteriormente
     * alterada.
     */
    public void fechar() {
        fechado = true;
    }

    /**
     * Converte o contedo desta lista de problemas em uma {@linkplain String}
     * em formato JSON.
     * <br />
     * A {@linkplain String} resultante contm mapeamentos de cada um dos tipos
     * de problemas informados para um array contendo as descries dos
     * problemas.
     * @return Uma representao JSON deste objeto.
     */
    @Override
    public String toString() {
        // Se o resultado j estiver em cache, apenas o retorna.
        if (toStringCache != null) return toStringCache;

        // Inicia a criao da String resultante.
        StringBuilder sb = new StringBuilder("{");

        // Itera os tipos de problemas.
        boolean primeiraIteracao1 = true;
        for (Map.Entry<E, List<String>> e : problemas.entrySet()) {

            // Acrescenta o tipo do problema. Separa com uma vrgula do tipo
            // anterior caso no seja o primeiro.
            if (!primeiraIteracao1) sb.append(", ");
            sb.append("\"").append(jsonFormat(e.getKey().name().toLowerCase())).append("\": [");

            // Em cada tipo de problema, itera as descries dos problemas.
            boolean primeiraIteracao2 = true;
            for (String p : e.getValue()) {

                // Acrescenta a descrio do problema. Separa com uma vrgula da
                // descrio anterior caso no seja a primeira.
                if (!primeiraIteracao2) sb.append(", ");
                sb.append('\"').append(jsonFormat(p)).append('\"');
                primeiraIteracao2 = false;
            }

            sb.append(']');
            primeiraIteracao1 = false;
        }
        sb.append('}');

        // Salva o resultado e retorna.
        return toStringCache = sb.toString();
    }

    /**
     * Retorna o tipo da lista de problemas.
     * @return O tipo da lista de problemas.
     */
    public Class<E> getTipo() {
        return classe;
    }

    /**
     * Lista os problemas de um determinado tipo.
     * @param tipo O tipo dos problemas a serem informados.
     * @return Os problemas correspondentes ao tipo especificado.
     * @throws ClassCastException Se o parmetro tipo no corresponder ao tipo
     * desta classe.
     */
    public List<String> getProblemas(E tipo) {
        verificarTipoValido(tipo);
        return Collections.unmodifiableList(problemas.get(tipo));
    }
}
