/*
 * 07/02/2009, 18:37.
 *
 * Simuquiz - http://www.simuquiz.com.br
 */
package br.com.simuquiz.entidades;

import br.com.simuquiz.conexao.Database;
import br.com.simuquiz.util.JsonMarshal;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.persistence.Entity;
import javax.persistence.EntityManager;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.NoResultException;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Query;

/**
 * Representa um usurio cadastrado do sistema.
 * @author Victor Williams Stafusa da Silva
 * @author Andr Luiz Pereira lvares
 */
@Entity
@JsonMarshal
public class Usuario implements Serializable {

    private static final long serialVersionUID = 1850067683492244417L;

    /** Chave primria do usurio. */
    @Id
    @GeneratedValue
    private long idUsuario;

    /** Nome do usurio. */
    private String nome;

    /** Login do usurio. */
    private String login;

    /** Senha do usurio. */
    private String senha;

    /** E-mail do usurio. */
    private String email;

    //@Lob
    //@Basic(fetch=FetchType.LAZY)
    //private byte[] foto;

    //@ManyToMany(mappedBy="Moderacao")
    //private Collection<Exame> examesModerados;

    @OneToMany(mappedBy="proprietario")
    @OrderBy(value="nome")
    private Collection<Exame> examesDono;

    @OneToMany(mappedBy="usuario")
    @OrderBy(value="inicio")
    private Collection<Prova> provasRealizadas;

    /**
     * Construtor nico.
     */
    public Usuario() {}

    @JsonMarshal
    public long getIdUsuario() {
        return idUsuario;
    }

    @JsonMarshal
    public String getNome() {
        return nome;
    }

    @JsonMarshal
    public String getLogin() {
        return login;
    }

    public String getSenha() {
        return senha;
    }

    @JsonMarshal
    public String getEmail() {
        return email;
    }

    public Exame novoExame(final String nome, final String descricaoCurta, final String descricaoCompleta) {
        return Database.efetuarTransacao(new Database.Transacao<Exame>() {

            @Override
            public Exame efetuarTransacao(EntityManager em) {
                Exame e = Exame.novo(em, Usuario.this, nome, descricaoCurta, descricaoCompleta);
                Usuario.this.examesDono.add(e);
                em.merge(Usuario.this);
                return e;
            }
        });
    }

    public Prova iniciarProva(final Exame exame) {
        return Database.efetuarTransacao(new Database.Transacao<Prova>() {
            @Override
            public Prova efetuarTransacao(EntityManager em) {
                Prova p = Prova.novo(em, Usuario.this, exame, exame.getQuestoes());
                provasRealizadas.add(p);
                em.merge(Usuario.this);
                return p;
            }
        });
    }

    public Collection<Prova> provasRealizadas() {
        return provasRealizadas;
    }

    @SuppressWarnings("unchecked")
    public Collection<Exame> listarExamesPossuidos(int primeiro, int quantidade) {
        Query query = Database.getEntityManager().createQuery("select e from Exame e where e.proprietario = :proprietario order by e.nome");
        query.setParameter("proprietario", this);
        query.setFirstResult(primeiro);
        query.setMaxResults(quantidade);
        return (Collection<Exame>) query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public Collection<MensagemPrivada> listarCaixaDeEntrada(int primeiro, int quantidade) {
        Query query = Database.getEntityManager().createQuery("select mp from MensagemPrivada mp where mp.destinatario = :usuario order by mp.dataEnvio");
        query.setParameter("usuario", this);
        query.setFirstResult(primeiro);
        query.setMaxResults(quantidade);
        return (Collection<MensagemPrivada>) query.getResultList();
    }

    @SuppressWarnings("unchecked")
    public Collection<MensagemPrivada> listarCaixaDeSaida(int primeiro, int quantidade) {
        Query query = Database.getEntityManager().createQuery("select mp from MensagemPrivada mp where mp.remetente = :usuario");
        query.setParameter("usuario", this);
        query.setFirstResult(primeiro);
        query.setMaxResults(quantidade);
        return (Collection<MensagemPrivada>) query.getResultList();
    }

    /**
     * Cadastra um novo usurio no banco de dados.
     * @param nome O nome do usurio.
     * @param login O login do usurio.
     * @param senha A senha do usurio.
     * @param email O endereo de e-mail do usurio.
     * @return O bean do usurio recm-cadastrado.
     */
    public static Usuario novo(String nome, String login, String senha, String email) {
        Usuario usuario = new Usuario();
        usuario.nome = nome;
        usuario.login = login;
        usuario.senha = senha;
        usuario.email = email;
        usuario.examesDono = new ArrayList<Exame>();
        usuario.provasRealizadas = new ArrayList<Prova>();

        Database.persist(usuario);
        return usuario;
    }

    /**
     * Atualiza um usurio no banco de dados.
     * @param nome O nome do usurio.
     * @param login O login do usurio.
     * @param senha A senha do usurio.
     * @param email O endereo de e-mail do usurio.
     */
    public void atualizar(String nome, String login, String senha, String email) {
        this.nome = nome;
        this.login = login;
        this.senha = senha;
        this.email = email;

    	Database.merge(this);
    }

    /**
     * Procura um usurio no banco de dados que tenha o login e a senha
     * informados.
     * @param login O login do usurio.
     * @param senha A senha do usurio.
     * @return O bean do usurio correspondente, ou {@code null} se no existir
     * usurio com o login e a senha informados.
     */
    public static Usuario login(String login, String senha) {
        Query query = Database.getEntityManager().createQuery("select u from Usuario u where login = :login and senha = :senha");
        query.setParameter("login", login);
        query.setParameter("senha", senha);
        try {
            return (Usuario) query.getSingleResult();
        } catch (NoResultException e) {
            return null;
        }
    }

    /**
     * Procura um usurio no banco de dados que tenha o login informados.
     * @param login O login do usurio.
     * @return O bean do usurio correspondente, ou {@code null} se no existir
     * usurio com o login informado.
     */
    public static Usuario pesquisarPorLogin(String login) {
        Query query = Database.getEntityManager().createQuery("select u from Usuario u where login = :login");
        query.setParameter("login", login);
        try {
            return (Usuario) query.getSingleResult();
        } catch (NoResultException e) {
            return null;
        }
    }

    /**
     * Procura um usurio no banco de dados com a chave primria informada.
     * @param id A chave primria.
     * @return O bean do usurio correspondente, ou {@code null} se no existir
     * usurio com a chave informada.
     */
    public static Usuario pesquisarPorId(long id) {
        return Database.getEntityManager().find(Usuario.class, id);
    }

    @SuppressWarnings("unchecked")
    public static List<Usuario> listarTudo() {
        Query q = Database.getEntityManager().createQuery("from Usuario");
        return (List<Usuario>) q.getResultList();
    }
}
