/*******************************************************************************
 ** BonnMotion - a mobility scenario generation and analysis tool             **
 ** Copyright (C) 2002-2005 University of Bonn                                **
 **                                                                           **
 ** This program is free software; you can redistribute it and/or modify      **
 ** it under the terms of the GNU General Public License as published by      **
 ** the Free Software Foundation; either version 2 of the License, or         **
 ** (at your option) any later version.                                       **
 **                                                                           **
 ** This program is distributed in the hope that it will be useful,           **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of            **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             **
 ** GNU General Public License for more details.                              **
 **                                                                           **
 ** You should have received a copy of the GNU General Public License         **
 ** along with this program; if not, write to the Free Software               **
 ** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA **
 *******************************************************************************/

package edu.bonn.cs.iv.bonnmotion.run;

import edu.bonn.cs.iv.bonnmotion.App;
import edu.bonn.cs.iv.bonnmotion.BMHelper;
import edu.bonn.cs.iv.bonnmotion.Model;

/** Frontend for all applications and scenario generators. */

public class BM {

	private final static String PROG_NAME = "BonnMotion";
	private final static String PROG_VER = "1.3a";
	private String fSettings = null;
	private String fSaveScenario = null;

	private void printHeader() {
		System.out.println(PROG_NAME + " " + PROG_VER + "\n");
	}

	private void printModules() {
		System.out.println("Available models: ");
		for (int i = 0; i < BMHelper.MODELS.length; i += 2)
			System.out.println("+ " + BMHelper.MODELS[i] + " - "
					+ BMHelper.MODELS[i + 1]);
	}

	private void printApps() {
		System.out.println("Available apps: ");
		for (int i = 0; i < BMHelper.APPS.length; i += 2)
			System.out.println("+ " + BMHelper.APPS[i] + " - "
					+ BMHelper.APPS[i + 1]);
	}

	private void printSpecificHelp(String _m) {
		Class c = BMHelper.str2class(_m);
		if (c == null) {
			System.out.println(_m + ": unknown");
			printModules();
		} else {
			try {
				c.getMethod("printHelp").invoke(null, null);
			} catch (Exception e) {
				App.exceptionHandler("could not print help to " + c, e);
			}
		}
	}

	public static void printHelp() {
		System.out.println("Help:");
		System.out.println("  -h                    	Print this help");
		System.out.println("");
		System.out.println("Scenario generation:");
		System.out
				.println("  -f <scenario name> [-I <parameter file>] <model name> [model options]");
		System.out
				.println("  -hm                           Print available models");
		System.out
				.println("  -hm <module name>             Print help to specific model");
		System.out.println("");
		System.out.println("Application:");
		System.out.println("  <application name> [Application-Options]");
		System.out
				.println("  -ha                           Print available applications");
		System.out
				.println("  -ha <application name>        Print help to specific application");
	}

	protected boolean parseArg(char key, String val) {
		switch (key) {
		case 'I':
			fSettings = val;
			return true;
		case 'f':
			fSaveScenario = val;
			return true;
		default:
			return false;
		}
	}

	/**
	 * Starts the magic. Determines the class to run specified on the command
	 * line and passes the parameters to the application or model.
	 */
	public void go(String[] _args) throws Throwable {
		printHeader();

		if (_args.length == 0) {
			printHelp();
		} else {

			// Get options: Is help needed?
			if (_args[0].equals("-h"))
				printHelp();
			else if (_args[0].equals("-hm"))
				if (_args.length > 1)
					printSpecificHelp(_args[1]);
				else
					printModules();
			else if (_args[0].equals("-ha"))
				if (_args.length > 1)
					printSpecificHelp(_args[1]);
				else
					printApps();
			else {

				int pos = 0;
				while ((_args[pos].charAt(0) == '-') || (pos == _args.length)) {
					char key = _args[pos].charAt(1);
					String value;
					if (_args[pos].length() > 2)
						value = _args[pos].substring(2);
					else
						value = _args[++pos];
					if (!parseArg(key, value)) {
						System.out.println("warning: ignoring unknown key "
								+ key);
					}
					pos++;
				}

				System.out.println("Starting " + _args[pos] + " ...");

				Class c;

				if ((c = BMHelper.str2class(_args[pos])) != null) {
					try {
						if (c.getPackage().getName().equals(BMHelper.APPS_PACK)) {
							String[] args = BMHelper.removeFirstElements(_args,
									pos + 1);
							Class[] cType = { String[].class };
							/*
							 * System.out.print("Launching \"" + c +
							 * "\", args: "); for (int i = 0; i < args.length;
							 * i++) { System.out.print("[" + args[i] + "]"); }
							 * System.out.println("");
							 */
							Object[] cParam = { args };
							c.getConstructor(cType).newInstance(cParam);
						} else {
							if (fSaveScenario == null) {
								System.out
										.println("Refusing to create a scenario which will not be saved anyhow. (Use -f.)");
								System.exit(0);
							}
							String[] args = BMHelper.removeFirstElements(_args,
									pos);
							args[0] = fSettings;
							Class[] cType = { String[].class };
							Object[] cParam = { args };
							((Model) c.getConstructor(cType)
									.newInstance(cParam)).write(fSaveScenario);
						}
						// System.out.println( " done." );
					} catch (ClassCastException e1) {
						System.out.println("ClassCastException");
					} catch (Exception e) {
						e.printStackTrace();
						App.exceptionHandler("Error in " + _args[pos], e);
					}
				} else {
					System.out.println("unknown");
					printHelp();
				}

				System.out.println(_args[pos] + " done.");
			}
		}
	}

	public static void main(String[] args) throws Throwable {
		new BM().go(args);
	}

} // BM
